<?php

declare(strict_types=1);

namespace ComerciaGlobalPayments\AddonPayments\SDK\_3DSecure\v2;

use ComerciaGlobalPayments\AddonPayments\SDK\_3DSecure\v2\Enum as _Enum;

/**
 * Class CardHolderAccountInfo.
 *
 * Models card holder account data for the 3DSecureV2 schema.
 *
 * @SuppressWarnings(PHPMD.LongVariable)
 */
class CardHolderAccountInfo extends AbstractSchemaObject
{
    /**
     * Property constants.
     */
    public const ACCOUNT_AGE = 'accountAge';

    public const ACCOUNT_CHANGE_DATE = 'accountChangeDate';

    public const ACCOUNT_CREATION_DATE = 'accountCreationDate';

    public const PASSWORD_CHANGE_DATE = 'passwordChangeDate';

    public const PAYMENT_ACCOUNT_AGE = 'paymentAccountAge';

    public const PROVISIONING_ATTEMPTS_DAY = 'provisioningAttemptsDay';

    public const PURCHASES_LAST_6_MONTHS = 'purchasesLast6Months';

    public const SHIPPING_ADDRESS_USAGE = 'shippingAddressUsage';

    public const SHIPPING_NAME_IND = 'shippingNameInd';

    public const SUSPICIOUS_ACCOUNT_ACTIVITY_IND = 'suspiciousAccountActivityInd';

    public const TXN_ACTIVITY_DAY = 'txnActivityDay';

    public const TXN_ACTIVITY_YEAR = 'txnActivityYear';

    /**
     * @var string Length of time that the cardholder has had the account
     */
    private $accountAge;

    /**
     * @var string|null Date that the cardholder’s account with the Merchant was last changed formatted as Y-m-d
     */
    private $accountChangeDate;

    /**
     * @var string|null Date that the cardholder opened the account with the Merchant formatted as Y-m-d
     */
    private $accountCreationDate;

    /**
     * @var string|null Date that cardholder’s account with the Merchant had a password change or account reset
     *                  formatted as Y-m-d
     */
    private $passwordChangeDate;

    /**
     * @var string|null Date that the payment account was enrolled in the cardholder’s account with the Merchant
     *                  formatted as Y-m-d
     */
    private $paymentAccountAge;

    /**
     * @var int|null Number of Add Card attempts in the last 24 hours, max-value 999
     */
    private $provisioningAttemptsDay;

    /**
     * @var int|null Number of purchases with this cardholder account during the previous six months, max-value 999
     */
    private $purchasesLast6Months;

    /**
     * @var string|null Date when the shipping address used for this transaction was first used with the Merchant
     *                  formatted as Y-m-d
     */
    private $shippingAddressUsage;

    /**
     * @var string Indicates if the Cardholder Name on the account is identical to the shipping Name used for this
     *                  transaction
     */
    private $shippingNameInd;

    /**
     * @var string Indicates whether the Merchant has experienced suspicious activity (including previous fraud)
     *                  on the cardholder account
     */
    private $suspiciousAccountActivityInd;

    /**
     * @var int|null Trx attempts in the last 24h (sucessfull or abandoned transactions count), max-value 999
     */
    private $txnActivityDay;

    /**
     * @var int|null Trx attempts in the last Year (sucessfull or abandoned transactions count), max-value 999
     */
    private $txnActivityYear;

    public function setAccountAge(string $accountAge): self
    {
        $this->accountAge = (string) (new _Enum\AccountAgeEnum($accountAge));

        return $this;
    }

    public function setAccountChangeDate(?string $accountChangeDate): self
    {
        $this->accountChangeDate = $accountChangeDate;

        return $this;
    }

    public function setAccountCreationDate(?string $accountCreationDate): self
    {
        $this->accountCreationDate = $accountCreationDate;

        return $this;
    }

    public function setPasswordChangeDate(?string $passwordChangeDate): self
    {
        $this->passwordChangeDate = $passwordChangeDate;

        return $this;
    }

    public function setPaymentAccountAge(?string $paymentAccountAge): self
    {
        $this->paymentAccountAge = $paymentAccountAge;

        return $this;
    }

    public function setProvisioningAttemptsDay(int $provisioningAttemptsDay = 0): self
    {
        $this->provisioningAttemptsDay = $provisioningAttemptsDay;

        return $this;
    }

    public function setPurchasesLast6Months(int $purchasesLast6Months = 0): self
    {
        $this->purchasesLast6Months = $purchasesLast6Months;

        return $this;
    }

    public function setShippingAddressUsage(?string $shippingAddressUsage): self
    {
        $this->shippingAddressUsage = $shippingAddressUsage;

        return $this;
    }

    public function setShippingNameInd(string $shippingNameInd): self
    {
        $this->shippingNameInd = (string) (new _Enum\ShippingNameIndEnum($shippingNameInd));

        return $this;
    }

    public function setSuspiciousAccountActivityInd(string $suspiciousAccountActivityInd): self
    {
        $this->suspiciousAccountActivityInd = (string) (new _Enum\SuspiciousAccountActivityIndEnum(
            $suspiciousAccountActivityInd
        ));

        return $this;
    }

    public function setTxnActivityDay(int $txnActivityDay = 0): self
    {
        $this->txnActivityDay = $txnActivityDay;

        return $this;
    }

    public function setTxnActivityYear(int $txnActivityYear = 0): self
    {
        $this->txnActivityYear = $txnActivityYear;

        return $this;
    }

    public function getAccountAge(): string
    {
        return $this->accountAge;
    }

    public function getAccountChangeDate(): ?string
    {
        return $this->accountChangeDate;
    }

    public function getAccountCreationDate(): ?string
    {
        return $this->accountCreationDate;
    }

    public function getPasswordChangeDate(): ?string
    {
        return $this->passwordChangeDate;
    }

    public function getPaymentAccountAge(): ?string
    {
        return $this->paymentAccountAge;
    }

    public function getProvisioningAttemptsDay(): ?int
    {
        return $this->provisioningAttemptsDay;
    }

    public function getPurchasesLast6Months(): ?int
    {
        return $this->purchasesLast6Months;
    }

    public function getShippingAddressUsage(): ?string
    {
        return $this->shippingAddressUsage;
    }

    public function getShippingNameInd(): string
    {
        return $this->shippingNameInd;
    }

    public function getSuspiciousAccountActivityInd(): string
    {
        return $this->suspiciousAccountActivityInd;
    }

    public function getTxnActivityDay(): ?int
    {
        return $this->txnActivityDay;
    }

    public function getTxnActivityYear(): ?int
    {
        return $this->txnActivityYear;
    }

    public function getName(): string
    {
        return 'cdAccountInfo3ds2';
    }

    public function getDefaults(): array
    {
        return [
            self::ACCOUNT_AGE => (string) (new _Enum\AccountAgeEnum()),
            self::ACCOUNT_CHANGE_DATE => null,
            self::ACCOUNT_CREATION_DATE => null,
            self::PASSWORD_CHANGE_DATE => null,
            self::PAYMENT_ACCOUNT_AGE => null,
            self::PROVISIONING_ATTEMPTS_DAY => 0,
            self::PURCHASES_LAST_6_MONTHS => 0,
            self::SHIPPING_ADDRESS_USAGE => null,
            self::SHIPPING_NAME_IND => (string) (new _Enum\ShippingNameIndEnum()),
            self::SUSPICIOUS_ACCOUNT_ACTIVITY_IND => (string) (new _Enum\SuspiciousAccountActivityIndEnum()),
            self::TXN_ACTIVITY_DAY => 0,
            self::TXN_ACTIVITY_YEAR => 0,
        ];
    }
}
