<?php

declare(strict_types=1);

namespace ComerciaGlobalPayments\AddonPayments\SDK\Request;

use ComerciaGlobalPayments\AddonPayments\SDK\Request\Enum\OperationTypeEnum;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * Class Authorize.
 *
 * Maps data for 'authorization' request.
 *
 * @see https://api-developer.easypaymentgateway.com/EPGJS/#operation/auth
 */
class Authorization extends AbstractRequest
{
    /**
     * @var string
     * @Assert\NotNull()
     */
    private $merchantId;

    /**
     * @var string
     * @Assert\NotNull()
     */
    private $merchantKey;

    /**
     * @var string
     * @Assert\NotNull()
     */
    private $customerId;

    /**
     * @var int
     * @Assert\NotNull()
     * @Assert\Type(
     *     type="integer",
     *     message="The value {{ value }} is not a valid {{ type }}."
     * )
     */
    private $productId;

    /**
     * @var string
     * @Assert\NotNull()
     * @Assert\Country()
     */
    private $country;

    /**
     * @var string
     * @Assert\NotNull()
     * @Assert\Currency()
     */
    private $currency;

    /**
     * @var string[]
     * @Assert\NotNull()
     * @Assert\Choice(
     *     callback="getOperationChoices",
     *     min=1,
     *     max=2,
     *     multiple=true,
     *     strict=true,
     *     minMessage="You must specify at least one operation",
     *     maxMessage="You cannot specify more than {{ limit }} operations",
     *     message="The value you selected is not a valid operation",
     *     multipleMessage="One or more of the given values is not a valid operation"
     * )
     */
    private $operations;

    /**
     * @var bool
     * @Assert\Type(
     *     type="bool",
     *     message="The value {{ value }} is not a valid {{ type }}."
     * )
     */
    private $anonymousCustomer = false;

    public function setMerchantId(string $merchantId): self
    {
        $this->merchantId = $merchantId;

        return $this;
    }

    public function getMerchantId(): string
    {
        return $this->merchantId;
    }

    public function setMerchantKey(string $merchantKey): self
    {
        $this->merchantKey = $merchantKey;

        return $this;
    }

    public function getMerchantKey(): string
    {
        return $this->merchantKey;
    }

    public function setCustomerId(string $customerId): self
    {
        $this->customerId = $customerId;

        return $this;
    }

    public function setProductId(int $productId): self
    {
        $this->productId = $productId;

        return $this;
    }

    public function setCountry(string $country): self
    {
        $this->country = $country;

        return $this;
    }

    public function setCurrency(string $currency): self
    {
        $this->currency = $currency;

        return $this;
    }

    public function setOperations(string $operation): self
    {
        $this->operations = [(string) (new OperationTypeEnum($operation))];

        return $this;
    }

    public function setAnonymousCustomer(bool $anonymousCustomer): self
    {
        $this->anonymousCustomer = $anonymousCustomer;

        return $this;
    }

    public function getCustomerId(): string
    {
        return $this->customerId;
    }

    public function getProductId(): int
    {
        return $this->productId;
    }

    public function getCountry(): string
    {
        return $this->country;
    }

    public function getCurrency(): string
    {
        return $this->currency;
    }

    /**
     * @return string[]
     */
    public function getOperations(): array
    {
        return $this->operations;
    }

    /**
     * @return string[]
     */
    public function getOperationChoices(): array
    {
        return array_values((new OperationTypeEnum())->getConstList());
    }

    public function isAnonymousCustomer(): bool
    {
        return $this->anonymousCustomer;
    }
}
