<?php

declare(strict_types=1);

namespace ComerciaGlobalPayments\AddonPayments\SDK\Request;

use ComerciaGlobalPayments\AddonPayments\SDK\Request\Enum\ChargeTypeEnum;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * Class ExpressCheckout.
 *
 * Maps data for express checkout request.
 */
class ExpressCheckout extends AbstractRequest
{
    /**
     * @var string
     * @Assert\NotNull()
     */
    private $merchantId;

    /**
     * @var string
     * @Assert\NotNull()
     * @Assert\Country()
     */
    private $country;

    /**
     * @var string
     * @Assert\NotNull()
     * @Assert\Currency()
     */
    private $currency;

    /**
     * @var string
     * @Assert\NotNull()
     * @Assert\Choice(
     *     callback="getOperationChoices",
     *     multiple=false,
     *     strict=true,
     *     message="The value you selected is not a valid operation",
     *     multipleMessage="One or more of the given values is not a valid operation"
     * )
     */
    private $operationType;

    /**
     * @var int
     * @Assert\NotNull()
     * @Assert\Type(
     *     type="integer",
     *     message="The value {{ value }} is not a valid {{ type }}."
     * )
     */
    private $productId;

    public function getMerchantId(): string
    {
        return $this->merchantId;
    }

    public function setMerchantId(string $merchantId): self
    {
        $this->merchantId = $merchantId;

        return $this;
    }

    public function getCountry(): string
    {
        return $this->country;
    }

    public function setCountry(string $country): self
    {
        $this->country = $country;

        return $this;
    }

    public function getCurrency(): string
    {
        return $this->currency;
    }

    public function setCurrency(string $currency): self
    {
        $this->currency = $currency;

        return $this;
    }

    public function getOperationType(): string
    {
        return $this->operationType;
    }

    public function setOperationType(string $operationType): self
    {
        $this->operationType = (string) (new ChargeTypeEnum($operationType));

        return $this;
    }

    public function getProductId(): int
    {
        return $this->productId;
    }

    public function setProductId(int $productId): self
    {
        $this->productId = $productId;

        return $this;
    }

    /**
     * @return string[]
     */
    public function getOperationChoices(): array
    {
        return array_values((new ChargeTypeEnum())->getConstList());
    }
}
