<?php
/**
 * 2007-2021 PrestaShop
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/afl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to http://www.prestashop.com for more information.
 *
 * @author    PrestaShop SA <contact@prestashop.com>
 * @copyright 2007-2021 PrestaShop SA
 * @license   http://opensource.org/licenses/afl-3.0.php  Academic Free License (AFL 3.0)
 *  International Registered Trademark & Property of PrestaShop SA
 */

namespace Prestashop\Module\AddonPayments\Operations;

use addonpayments;
use ComerciaGlobalPayments\AddonPayments\SDK\Api\EPGJs;
use ComerciaGlobalPayments\AddonPayments\SDK\Request\Authorization;
use ComerciaGlobalPayments\AddonPayments\SDK\Request\Enum\OperationTypeEnum;
use ComerciaGlobalPayments\AddonPayments\SDK\Response\ErrorResponseInterface;
use Configuration;
use Exception;
use Prestashop\Module\AddonPayments\AddonPaymentsConfig;
use Prestashop\Module\AddonPayments\Exception\CheckoutException;
use Prestashop\Module\AddonPayments\Services\ApiFactory;
use PrestaShopLogger;

/**
 * Performs an authorization request through EPGJs API.
 */
final class Authorize
{
    public function __invoke(
        $customerId,
        string $currencyCode,
        string $countryCode,
        bool $isGuest = true,
        string $operationType = OperationTypeEnum::DEPOSIT,
        string $productId = null
    ): ?string {
        try {
            /** @var \ComerciaGlobalPayments\AddonPayments\SDK\Api\EPGJs $client */
            $client = ApiFactory::createInstance(EPGJs::NAME);
            $request = (new Authorization())
                ->setCustomerId($customerId)
                ->setCurrency($currencyCode)
                ->setCountry($countryCode)
                ->setProductId(
                    (int) ($productId ??
                        Configuration::get(AddonPaymentsConfig::EPG_PRODUCT_ID))
                )
                ->setOperations($operationType)
                ->setAnonymousCustomer($isGuest);
            /** @var \ComerciaGlobalPayments\AddonPayments\SDK\Response\Authorized|\ComerciaGlobalPayments\AddonPayments\SDK\Response\ErrorResponse $response */
            $response = $client->authorization($request);
            if ($response instanceof ErrorResponseInterface) {
                throw new CheckoutException($response->getReason(), CheckoutException::API_AUTHORIZATION_FAILED);
            }

            return $response->getAuthToken();
        } catch (CheckoutException $e) {
            PrestaShopLogger::addLog('Authorization failed: ' . $e->getMessage(), 3);
        } catch (Exception $e) {
            PrestaShopLogger::addLog($e->getMessage(), 4);
        }

        return null;
    }
}
