<?php
/**
 * 2007-2021 PrestaShop
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/afl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to http://www.prestashop.com for more information.
 *
 * @author    PrestaShop SA <contact@prestashop.com>
 * @copyright 2007-2021 PrestaShop SA
 * @license   http://opensource.org/licenses/afl-3.0.php  Academic Free License
 *            (AFL 3.0) International Registered Trademark & Property of
 *            PrestaShop SA
 */

namespace Prestashop\Module\AddonPayments\Operations;

use ComerciaGlobalPayments\AddonPayments\SDK\Response\OperationInterface;
use Exception;
use Order;
use Prestashop\Module\AddonPayments\OperationContext;
use PrestaShopLogger;
use Transactions;

/**
 * Implements common features related wih payment refund operations.
 */
trait RefundPaymentsTrait
{
    /**
     * @var \Transactions[]
     */
    protected $transactions;

    protected function findTransaction(Order $order, $transaction_id): ?Transactions
    {
        if (!$this->transactions) {
            $this->transactions = $this->getRefundableTransactionsByOrder($order);
        }

        if (empty($this->transactions)) {
            PrestaShopLogger::addLog(
                "No refundable transactions found for order {$order->reference}",
                1,
                null,
                'Order',
                $order->id
            );

            return null;
        }

        foreach ($this->transactions as $transaction) {
            if ($transaction->payFrexTransactionId === (string) $transaction_id) {
                return $transaction;
            }
        }

        return null;
    }

    /**
     * Returns an array of refundable transactions for an order.
     *
     * A refundable transaction is a credit or debit transaction with an amount greater than 0, marked as SUCCESS
     * and that has not been voided, cancelled or refunded by any other transaction.
     *
     * @return \Transactions[] returns an empty array if the previous conditions are not met
     */
    protected function getRefundableTransactionsByOrder(Order $order): array
    {
        try {
            $transactions = Transactions::getRefundableByOrder($order->id);
            if (empty($transactions)) {
                return [];
            }

            $refundable = [];
            foreach ($transactions as $transaction) {
                $refund = Transactions::getRemainingAmountByTransaction($transaction->payFrexTransactionId);
                if (false === $refund
                    || ($refund > 0 && $refund <= $transaction->amount)
                ) {
                    $transaction->remainingAmount = $transaction->amount - (float) $refund;
                    $refundable[$transaction->id_transaction] = $transaction;
                }
            }

            return $refundable;
        } catch (Exception $e) {
            PrestaShopLogger::addLog($e->getMessage(), 3, $e->getCode(), 'Order', $order->id);
        }

        return [];
    }

    private function processTransactions(OperationInterface $response, OperationContext $context): void
    {
        $operations = $response->count();
        foreach ($response->getOperations() as $transaction) {
            (new ProcessTransaction($context, $this->module))(
                $transaction,
                $transaction->getSortedOrder() === $operations
            );
        }
    }
}
