<?php
/**
 * 2007-2021 PrestaShop
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/afl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to http://www.prestashop.com for more information.
 *
 * @author    PrestaShop SA <contact@prestashop.com>
 * @copyright 2007-2021 PrestaShop SA
 * @license   http://opensource.org/licenses/afl-3.0.php  Academic Free License (AFL 3.0)
 *  International Registered Trademark & Property of PrestaShop SA
 */

namespace Prestashop\Module\AddonPayments;

use addonpayments;
use Configuration;
use ComerciaGlobalPayments\AddonPayments\SDK\Api\EPGJs;
use ComerciaGlobalPayments\AddonPayments\SDK\Encryption\Cypher\Aes256Cbc;

class AddonPaymentsConfig
{

    public const LIVE_MODE = 'ADDONPAYMENTS_LIVE_MODE';

    public const POST_INTEGRITY_CHECK = 'ADDONPAYMENTS_POST_INTEGRITY_CHECK';

    public const EPG_STYLE = 'ADDONPAYMENTS_EPG_STYLE';

    public const PRINTED_KEY = 'ADDONPAYMENTS_PRINTED_KEY';

    public const ENCRYPTION_MODE = 'ADDONPAYMENTS_ENCRYPTION_MODE';

    // Default configuration - Payment title and logo
    public const DEFAULT_CONFIGURATION = 'ADDONPAYMENTS_PAYMENT_CONFIGURATION';

    public const TITLE = 'ADDONPAYMENTS_PAYMENT_TITLE';

    public const LOGO = 'ADDONPAYMENTS_PAYMENT_LOGO';

    public const DEFAULT_TITLE = 'ADDONPAYMENTS_PAYMENT_DEFAULT_TITLE';

    public const DEFAULT_LOGO = 'ADDONPAYMENTS_PAYMENT_DEFAULT_LOGO';

    public const DEFAULT_OPTION_CREDIT_CARD = 'credit_card';

    public const DEFAULT_OPTION_CREDIT_CARD_BIZUM = 'credit_card_bizum';

    public const DEFAULT_OPTION_QUIX = 'quix';

    public const DEFAULT_OPTION_MULTIPLE = 'multiple';

    public const DEFAULT_OPTION_GLOBALPAYMENTS = 'globalpayments';

    public const DEFAULT_OPTION_CUSTOM = 'custom';

    //Merchant
    public const MERCHANT_ID = 'ADDONPAYMENTS_MERCHANT_ID';

    public const MERCHANT_PASSWORD = 'ADDONPAYMENTS_MERCHANT_PASSWORD';

    public const EPG_PRODUCT_ID = 'ADDONPAYMENTS_EPG_PRODUCT_ID';

    public const MERCHANT_KEY = 'ADDONPAYMENTS_MERCHANT_KEY';

    public const SETTLE = 'ADDONPAYMENTS_SETTLE';

    // Financing
    public const FINANCING_OPTIONS = 'ADDONPAYMENTS_FINANCING_OPTIONS';

    public const BANNER_STYLE_COLOR = 'ADDONPAYMENTS_BANNER_STYLE_COLOR';

    public const BANNER_POSITION = 'ADDONPAYMENTS_BANNER_POSITION';

    public const BANNER_STYLE_BRANDING = 'ADDONPAYMENTS_BANNER_STYLE_BRANDING';

    public const BANNER_STYLE_VARIANT = 'ADDONPAYMENTS_BANNER_STYLE_VARIANT';

    public const FINANCING_DEFAULT_OPTION = 'instalments';

    // Callback urls
    public const CANCEL_URL = 'ADDONPAYMENTS_CANCEL_URL';

    public const FAILURE_URL = 'ADDONPAYMENTS_FAILURE_URL';

    public const SUCCESS_URL = 'ADDONPAYMENTS_SUCCESS_URL';


    /**
     * @var \addonpayments
     */
    private $module;

    /**
     * @param \addonpayments $module
     */
    public function __construct(addonpayments $module)
    {
        $this->module = $module;
    }

    /**
     * @return array
     */
    public function getDefaultPaymentConfig()
    {
        return [
            'logo_uri' => $this->getDefaultLogoUri(),
            self::DEFAULT_OPTION_CREDIT_CARD => $this->module->l('Credit Card', 'AddonPaymentsConfig'),
            self::DEFAULT_OPTION_CREDIT_CARD_BIZUM => $this->module->l('Credit Card or Bizum', 'AddonPaymentsConfig'),
            self::DEFAULT_OPTION_QUIX => $this->module->l('Pay in instalments', 'AddonPaymentsConfig'),
            self::DEFAULT_OPTION_MULTIPLE => $this->module->l('Select your payment method', 'AddonPaymentsConfig'),
            self::DEFAULT_OPTION_GLOBALPAYMENTS => $this->module->l('Select your payment method', 'AddonPaymentsConfig'),
            self::DEFAULT_OPTION_CUSTOM => $this->module->l('Custom', 'AddonPaymentsConfig'),
        ];
    }

    /**
     * @param $payment_option
     *
     * @return string
     */
    public function getDefaultTitle($payment_option)
    {
        return $this->getDefaultPaymentConfig()[$payment_option];
    }

    /**
     * @return string
     */
    public function getDefaultLogoUri()
    {
        return $this->module->getPathUri().'views'.DIRECTORY_SEPARATOR.'img'.DIRECTORY_SEPARATOR.'default-config'
               .DIRECTORY_SEPARATOR;
    }

    /**
     * @return string
     */
    public function getDefaultLogoLocalPath()
    {
        return $this->module->getLocalPath().'views'.DIRECTORY_SEPARATOR.'img'.DIRECTORY_SEPARATOR.'default-config'
               .DIRECTORY_SEPARATOR;
    }

    /**
     * Save all module configuration
     *
     * @return void
     */
    public function install()
    {
        Configuration::updateValue(self::LIVE_MODE, false);
        Configuration::updateValue(self::ENCRYPTION_MODE, Aes256Cbc::NAME);
        Configuration::updateValue(self::EPG_STYLE, EPGJs::EPGJS_STYLE_BOOTSTRAP);
        Configuration::updateValue(self::POST_INTEGRITY_CHECK, sha1(base64_encode(openssl_random_pseudo_bytes(32)).$this->module->name.time()));
        Configuration::updateValue(self::PRINTED_KEY, true);

        Configuration::updateValue(self::DEFAULT_CONFIGURATION, '');
        Configuration::updateValue(self::TITLE, '');
        Configuration::updateValue(self::LOGO, '');

        //Merchant
        Configuration::updateValue(self::MERCHANT_ID, null);
        Configuration::updateValue(self::MERCHANT_KEY, null);
        Configuration::updateValue(self::MERCHANT_PASSWORD, null);
        Configuration::updateValue(self::EPG_PRODUCT_ID, null);
        Configuration::updateValue(self::SETTLE, true);

        //Financing
        Configuration::updateValue(self::FINANCING_OPTIONS, 'instalments');
        Configuration::updateValue(self::BANNER_POSITION, 1);
        Configuration::updateValue(self::BANNER_STYLE_COLOR, '#34AEEB');
        Configuration::updateValue(self::BANNER_STYLE_VARIANT, 'text_banner');
        Configuration::updateValue(self::BANNER_STYLE_BRANDING, 'lending_hub');

        //Callback urls
        Configuration::updateValue(self::SUCCESS_URL, null);
        Configuration::updateValue(self::FAILURE_URL, null);
        Configuration::updateValue(self::CANCEL_URL, null);


    }

    /**
     * Remove all module configuration
     *
     * @return void
     */
    public function uninstall()
    {
        Configuration::deleteByName(self::LIVE_MODE);
        Configuration::deleteByName(self::ENCRYPTION_MODE);
        Configuration::deleteByName(self::EPG_STYLE);
        Configuration::deleteByName(self::POST_INTEGRITY_CHECK);
        Configuration::deleteByName(self::PRINTED_KEY);

        // Payment Method
        Configuration::deleteByName(self::DEFAULT_CONFIGURATION);
        Configuration::deleteByName(self::TITLE);
        Configuration::deleteByName(self::LOGO);

        // Merchant
        Configuration::deleteByName(self::MERCHANT_ID);
        Configuration::deleteByName(self::MERCHANT_KEY);
        Configuration::deleteByName(self::MERCHANT_PASSWORD);
        Configuration::deleteByName(self::EPG_PRODUCT_ID);
        Configuration::deleteByName(self::SETTLE);

        //Financing
        Configuration::deleteByName(self::FINANCING_OPTIONS);
        Configuration::deleteByName(self::BANNER_POSITION);
        Configuration::deleteByName(self::BANNER_STYLE_COLOR);
        Configuration::deleteByName(self::BANNER_STYLE_VARIANT);
        Configuration::deleteByName(self::BANNER_STYLE_BRANDING);

        //Callback urls
        Configuration::deleteByName(self::SUCCESS_URL);
        Configuration::deleteByName(self::FAILURE_URL);
        Configuration::deleteByName(self::CANCEL_URL);
    }
}
