'use strict';

var Logger = require('dw/system/Logger');

var Calendar = require('dw/util/Calendar');
var StringUtils = require('dw/util/StringUtils');

/**
 * The default date format is US.
 */
var DEFAULT_LOCALE_DATE_FORMAT = 'en_US';

/**
 * Get current date with current time zone.
 * https://documentation.b2c.commercecloud.salesforce.com/DOC1/index.jsp?topic=%2Fcom.demandware.dochelp%2FDWAPI%2Fscriptapi%2Fhtml%2Fapi%2Fclass_dw_util_Calendar.html
 * WARNING: Keep in mind that the returned Date object's time is always interpreted in the time zone GMT.
 *          This means time zone information set at the calendar object will not be honored and gets lost.
 * @returns {Date} - Current date with current time zone or default date (exception).
 */
function currentDate() {
    try {
        var calendar = new Calendar();
        calendar.setTimeZone(require('dw/system/Site').getCurrent().getTimezone());
        var formatCalendar = StringUtils.formatCalendar(calendar, DEFAULT_LOCALE_DATE_FORMAT, Calendar.INPUT_DATE_TIME_PATTERN);
        return new Date(formatCalendar);
    } catch (e) {
        Logger.error('Cannot get the current date: {0}', e);
    }
    return new Date();
}

/**
 * Format current date.
 * @param {string} pattern - The format pattern.
 * @returns {string | null} - Format current date or null.
 */
function formatCurrentDate(pattern) {
    try {
        return StringUtils.formatCalendar(new Calendar(currentDate()), pattern);
    } catch (e) {
        Logger.error('The current date cannot be formatted with the {0} pattern: {1}', pattern, e);
    }
    return null;
}

/**
 * Format date.
 * @param {Date} date - Date to check
 * @param {string} pattern - The format pattern.
 * @returns {string | null} - Format date or null.
 */
function formatDate(date, pattern) {
    try {
        return StringUtils.formatCalendar(new Calendar(date), pattern);
    } catch (e) {
        Logger.error('The {0} date cannot be formatted with the {1} pattern: {2}', date, pattern, e);
    }
    return null;
}

module.exports = {
    currentDate       : currentDate,
    formatCurrentDate : formatCurrentDate,
    formatDate        : formatDate
};
