'use strict';

var Logger = require('dw/system/Logger');

/**
 * Check is variable is null (this means not undefined and not null)
 * Avoid integer zero false positives
 * @param {Object} variable - an object to check
 * @return {boolean} - true if variable is not null
 */
function isNull(variable) {
    if (variable !== null && variable !== undefined) {
        return false;
    }
    return true;
}

/**
 * Check is variable is null or empty (this means not undefined, not null y and not empty)
 * Avoid integer zero false positives
 * @param {Object} variable - an object to check
 * @return {boolean} - true if variable is not null, undefined or empty
 */
function isEmpty(variable) {
    if (variable !== null && variable !== undefined && variable !== '') {
        return false;
    }
    return true;
}

/**
 * Check if the object is empty
 * @param {Object} obj - the target object
 * @returns {boolean} - true if the object is empty
 */
function isEmptyObject(obj) {
    for (var prop in obj) {
        if (obj.hasOwnProperty(prop)) {
            return false;
        }
    }
    return JSON.stringify(obj) === JSON.stringify({});
}

/**
 * Check when argument type matches right value (only primitives)
 * @param {Object} typeName - type to check with name object
 * @param {string} name - name of argument in string format
 * @param {string} value - value of 'name'
 * @returns {boolean} true if argument type is valid, else false
 */
function checkArgumentType(typeName, name, value) {
    // eslint-disable-next-line valid-typeof
    if (!(typeof typeName === value)) {
        Logger.warn('Invalid argument type {0} -> {1}. Expected {2} type', name, value, typeName);
        return false;
    }
    return true;
}

/**
 * Check when argument value has value
 * @param {string} name - name of argument in string format
 * @param {Object} value - value of 'name'
 * @returns {boolean} true if argument type is valid, else false
 */
function checkArgumentValue(name, value) {
    if (!value) {
        Logger.warn('Invalid argument value {0} -> null', name);
        return false;
    }
    return true;
}

/**
 * Check when dw type matches right value (SFCC types)
 * @param {Object} typeName - type to check with name object
 * @param {string} name - name of argument in string format
 * @param {Object} value - value of 'name'
 * @param {boolean} printLog - print log
 * @returns {boolean} true if argument type is valid, else false
 */
function checkDwType(typeName, name, value, printLog) {
    if (typeName !== undefined && typeName !== null && typeName.class === value) {
        return true;
    }
    if (printLog) {
        Logger.warn('Invalid argument type {0} -> {1}. Expected {2} type', name, value, typeName);
    }
    return false;
}

/**
 * Check when dw type matches right value (SFCC types) and has the custom attribute.
 * @param {Object} typeName - type to check with name object
 * @param {string} name - name of argument in string format
 * @param {Object} value - value of 'name'
 * @param {boolean} printLog - print log
 * @returns {boolean} true if argument type is valid, else false
 */
function checkDwCustomType(typeName, name, value, printLog) {
    if (typeName !== undefined && typeName !== null && typeName.class === value && 'custom' in typeName) {
        return true;
    }
    if (printLog) {
        Logger.warn('Invalid argument type {0} -> {1}. Expected {2} type', name, value, typeName);
    }
    return false;
}

/**
 * Check when argument value has value
 * @param {Object} type - The type
 * @param {string} typeName - The type name
 * @param {string} attr - The attribute name
 * @param {boolean} printLog - print log
 * @returns {boolean} true if type has the attribute, else false
 */
function checkTypeHasArgument(type, typeName, attr, printLog) {
    if (!type) {
        return false;
    }
    if (attr in type) {
        return true;
    }
    if (printLog) {
        Logger.warn('The type {0} does not have the attribute {1}', typeName, attr);
    }
    return false;
}

module.exports = {
    isNull               : isNull,
    isEmpty              : isEmpty,
    isEmptyObject        : isEmptyObject,
    checkArgumentType    : checkArgumentType,
    checkArgumentValue   : checkArgumentValue,
    checkDwType          : checkDwType,
    checkDwCustomType    : checkDwCustomType,
    checkTypeHasArgument : checkTypeHasArgument
};
