'use strict';

var base = module.superModule;

var Logger = require('dw/system/Logger');
var Resource = require('dw/web/Resource');
var URLUtils = require('dw/web/URLUtils');

/**
 * Stores the order number in session
 * @param {dw.order.Order} order - The order
 */
function setOrderNoInSession(order) {
    session.custom.orderNo = order.orderNo;
}

/**
 * Returns the order number in session
 * @returns {string} - The order number
 */
function getOrderNoInSession() {
    return session.custom.orderNo ? session.custom.orderNo.toString() : '';
}

/**
 * Returns the order token in session
 * @returns {string} - The order token
 */
function getOrderToInSession() {
    return session.custom.orderTo ? session.custom.orderTo.toString() : '';
}

/**
 * Deletes the order number in session
 */
function deleteOrderNoInSession() {
    session.custom.orderNo = null;
    session.custom.orderTo = null;
}

/**
 * Save custom attributes
 * In order to avoid issues with SFRA it is added this empty function (used in upper cartridges if it is needed)
 * @param {dw.order.Order} - The order
 * @returns {Object} - The result
 */
function saveCustomAttributes(order) {
    var Transaction = require('dw/system/Transaction');
    var paymentInstrumentHelper = require('*/cartridge/scripts/addonPayments/helpers/paymentInstrumentHelper');
    var result = {};

    if (!order) {
        result.error = true;
        return result;
    }

    // Set the order custom attributes REMIND Transaction.wrapper()
    Transaction.wrap(function () {
        var paymentInstrument = paymentInstrumentHelper.getAddonPaymentInstrument(order);
        if (paymentInstrument) {
            order.custom.addonPaymentMethod = 'ADDON_PAYMENTS';
            order.custom.ADDON_PAYMENTS_API_PaymentID = paymentInstrument.custom.addonPaymentID;
        }
    });

    result.error = false;
    return result;
}

/**
 * Validate the custom billing form
 * @param {Object} billingForm - The billing form
 * @returns {Array | null} - The form field errors or null
 */
function validateCustomBillingForm(billingForm) {
    var COHelpers = require('*/cartridge/scripts/checkout/checkoutHelpers');

    if (!billingForm) {
        return null;
    }

    var formFieldErrors = [];

    var billingFormErrors = COHelpers.validateBillingForm(billingForm.addressFields);
    if (Object.keys(billingFormErrors).length) {
        formFieldErrors.push(billingFormErrors);
    }

    var contactInfoFormErrors = COHelpers.validateFields(billingForm.contactInfoFields);
    if (Object.keys(contactInfoFormErrors).length) {
        formFieldErrors.push(contactInfoFormErrors);
    }

    return formFieldErrors;
}

/**
 * Save the billing address
 * @param {Object} billingForm - The billing form
 * @param {dw.order.Basket} currentBasket - The current basket
 * @returns {boolean} - Return true if the billing address is save
 */
function saveCustomBillingAddress(billingForm, currentBasket) {
    var Transaction = require('dw/system/Transaction');

    if (!billingForm || !currentBasket) {
        return false;
    }

    try {
        var billingAddress = currentBasket.billingAddress;
        Transaction.wrap(function () {
            if (!billingAddress) {
                billingAddress = currentBasket.createBillingAddress();
            }

            billingAddress.setFirstName(billingForm.addressFields.firstName.value);
            billingAddress.setLastName(billingForm.addressFields.lastName.value);
            billingAddress.setAddress1(billingForm.addressFields.address1.value);
            billingAddress.setAddress2(billingForm.addressFields.address2.value);
            billingAddress.setCity(billingForm.addressFields.city.value);
            billingAddress.setPostalCode(billingForm.addressFields.postalCode.value);
            if (Object.prototype.hasOwnProperty.call(billingForm.addressFields, 'states')) {
                billingAddress.setStateCode(billingForm.addressFields.states.stateCode.value);
            }
            billingAddress.setCountryCode(billingForm.addressFields.country.value);
            billingAddress.setPhone(billingForm.addressFields.phone.value);
        });
    } catch (e) {
        Logger.error('Cannot save the billing address: {0}', e);
        return false;
    }

    return true;
}

/**
 * Create the new order (CheckoutServices-PlaceOrder)
 * @param {dw.order.Basket} currentBasket - The current basket.
 * @returns {Object} - The result: the order number or error with the error message
 */
function createNewOrder(currentBasket) {
    var Transaction = require('dw/system/Transaction');
    var COHelpers = require('*/cartridge/scripts/checkout/checkoutHelpers');
    var validationHelpers = require('*/cartridge/scripts/helpers/basketValidationHelpers');
    var hooksHelper = require('*/cartridge/scripts/helpers/hooks');
    var basketCalculationHelpers = require('*/cartridge/scripts/helpers/basketCalculationHelpers');

    if (!currentBasket) {
        return {
            error       : true,
            redirectUrl : URLUtils.url('Cart-Show').toString()
        };
    }

    var validatedProducts = validationHelpers.validateProducts(currentBasket);
    if (validatedProducts.error) {
        return {
            error       : true,
            redirectUrl : URLUtils.url('Cart-Show').toString()
        };
    }

    var validationOrderStatus = hooksHelper('app.validate.order', 'validateOrder', currentBasket,
        require('*/cartridge/scripts/hooks/validateOrder').validateOrder);
    if (validationOrderStatus.error) {
        return {
            error        : true,
            errorMessage : validationOrderStatus.message
        };
    }

    // Check to make sure there is a shipping address
    if (currentBasket.defaultShipment.shippingAddress === null) {
        return {
            error      : true,
            errorStage : {
                stage : 'shipping',
                step  : 'address'
            },
            errorMessage: Resource.msg('error.no.shipping.address', 'checkout', null)
        };
    }

    // Check to make sure billing address exists
    if (!currentBasket.billingAddress) {
        return {
            error      : true,
            errorStage : {
                stage : 'payment',
                step  : 'billingAddress'
            },
            errorMessage: Resource.msg('error.no.billing.address', 'checkout', null)
        };
    }

    // Calculate the basket
    Transaction.wrap(function () {
        basketCalculationHelpers.calculateTotals(currentBasket);
    });

    // Re-calculate the payments.
    var calculatedPaymentTransactionTotal = COHelpers.calculatePaymentTransaction(currentBasket);
    if (calculatedPaymentTransactionTotal.error) {
        return {
            error        : true,
            errorMessage : Resource.msg('error.technical', 'checkout', null)
        };
    }

    // Creates a new order.
    var order = COHelpers.createOrder(currentBasket);
    if (!order) {
        return {
            error        : true,
            errorMessage : Resource.msg('error.technical', 'checkout', null)
        };
    }

    return {
        error : false,
        order : order
    };
}

/**
 * Check if exists customerNo and if exists customer save address in AddressBook
 * @param {dw.order.Order} order - The order
 */
function saveAddressCustomer(order) {
    var CustomerMgr = require('dw/customer/CustomerMgr');
    var addressHelpers = require('*/cartridge/scripts/helpers/addressHelpers');

    if (order && order.customerNo) {
        var customer = CustomerMgr.getCustomerByCustomerNumber(order.customerNo);
        if (customer && customer.addressBook) {
            // save all used shipping addresses to address book of the logged in customer
            var allAddresses = addressHelpers.gatherShippingAddresses(order);
            allAddresses.forEach(function (address) {
                if (!addressHelpers.checkIfAddressStored(address, customer.addressBook.addresses)) {
                    addressHelpers.saveAddress(address, customer, addressHelpers.generateAddressName(address));
                }
            });
        }
    }
}

/**
 * Check if checkout has a invalid customer
 * @param {string} xIsQueryString - The x-is-query_string header parameter
 * @param {string | null} email - The basket email
 * @returns {boolean} - Return true if checkout has a invalid customer
 */
function isInvalidCustomer(xIsQueryString, email) {
    return xIsQueryString && xIsQueryString !== 'registration=submitted' && !email;
}


base.setOrderNoInSession = setOrderNoInSession;
base.getOrderNoInSession = getOrderNoInSession;
base.getOrderToInSession = getOrderToInSession;
base.deleteOrderNoInSession = deleteOrderNoInSession;
base.saveCustomAttributes = saveCustomAttributes;
base.validateCustomBillingForm = validateCustomBillingForm;
base.saveCustomBillingAddress = saveCustomBillingAddress;
base.createNewOrder = createNewOrder;
base.saveAddressCustomer = saveAddressCustomer;
base.isInvalidCustomer = isInvalidCustomer;


module.exports = base;
