'use strict';

const Transaction = require('dw/system/Transaction');
const Money = require('dw/value/Money');
const PaymentMgr = require('dw/order/PaymentMgr');
const {
    addonPaymentMethodId
} = require('../../../config/addonPaymentsPreferences');

/**
 * Calculates the amount to be payed by a non-gift certificate payment instrument based
 * on the given basket. The method subtracts the amount of all redeemed gift certificates
 * from the order total and returns this value.
 *
 * @param {Object} lineItemCtnr - LineIteam Container (Basket or Order)
 * @returns {dw.value.Money} non gift certificate amount
 */
function calculateNonGiftCertificateAmount(lineItemCtnr) {
    let giftCertTotal = new Money(0.0, lineItemCtnr.currencyCode);
    const gcPaymentInstrs = lineItemCtnr.getGiftCertificatePaymentInstruments();
    const iter = gcPaymentInstrs.iterator();
    let orderPI = null;

    while (iter.hasNext()) {
        orderPI = iter.next();
        giftCertTotal = giftCertTotal.add(
            orderPI.getPaymentTransaction().getAmount()
        );
    }

    const orderTotal = lineItemCtnr.totalGrossPrice;
    const amountOpen = orderTotal.subtract(giftCertTotal);
    return amountOpen;
}

/**
 * Calculates the amount to be payed is equals than send to payment method
 *
 * @param {Object} lineItemCtnr - LineIteam Container (Basket or Order)
 * @param {Number} amount - Amount check
 * @returns {Boolean} If checkAmount is equals amount
 */
function checkTotalAmount(lineItemCtnr, amount) {
    const checkAmount =
    Math.round(calculateNonGiftCertificateAmount(lineItemCtnr).value * 100) /
    100;

    return checkAmount === amount;
}

/**
 * Return Addon Payments order payment instrument
 *
 * @param {dw.order.LineItemCtnr} order - Order
 * @returns {dw.order.OrderPaymentInstrument} payment instrument with id Addon Payments
 */
function getAddonPaymentInstrument(order) {
    const paymentInstruments = order.getPaymentInstruments(addonPaymentMethodId);
    return !empty(paymentInstruments) && paymentInstruments[0];
}

/**
 * Delete all Addon payment instruments from Order
 *
 * @param {dw.order.LineItemCtnr} order - Order
 */
function removeAddonPaymentInstrument(order) {
    const paymentInstrument = order.getPaymentInstruments(addonPaymentMethodId);
    if (empty(paymentInstrument)) return;
    Transaction.wrap(function () {
        order.removePaymentInstrument(paymentInstrument[0]);
    });
}

/**
 * CreatePaymentInstrument
 *
 * @param {Object} order - Order
 * @param {string} paymentType - Name of the payment method.
 * @returns {Object} Payment instrument
 */
function createPaymentInstrument(order, paymentType) {
    let paymentInstr = null;
    const paymentProcessor =
    PaymentMgr.getPaymentMethod(paymentType).getPaymentProcessor();

    Transaction.wrap(function () {
        paymentInstr = order.createPaymentInstrument(paymentType, order.totalGrossPrice);
        paymentInstr.paymentTransaction.setPaymentProcessor(paymentProcessor);
    });

    return paymentInstr;
}

/**
 * Removes payment instruments from Basket except GIFT_CERTIFICATE
 *
 * @param {Object} basket - Basket
 */
function removeNonAddonPaymentInstrument(basket) {
    const paymentInstruments = basket.getPaymentInstruments();

    const iterator = paymentInstruments.iterator();
    let paymentInstrument = null;
    Transaction.wrap(function () {
        while (iterator.hasNext()) {
            paymentInstrument = iterator.next();
            if (paymentInstrument.paymentMethod === 'GIFT_CERTIFICATE') {
                /* eslint-disable no-continue */
                continue;
            }
            basket.removePaymentInstrument(paymentInstrument);
        }
    });
}

/**
 * Update Refund Quix order status
 *
 * @param {string} orderNo - order number
 * @param {string} orderToken - orderToken
 * @param {string} infoRefund - info refund
 */
function updateQuixRefundTransaction(orderNo, orderToken, infoRefund){
    const OrderMgr = require('dw/order/OrderMgr');
    const Order = require('dw/order/Order');

    var order = OrderMgr.getOrder(orderNo, orderToken);
  
    if (order) {
      const paymentInstrument = getAddonPaymentInstrument(order);
      const Money = require('dw/value/Money');
      var totalRefunded = parseFloat(paymentInstrument.custom.addonPaymentRefund.slice(4));
  
      var paymentStatus = order.totalGrossPrice.getValue().toFixed(2) === (infoRefund.refunded_amount + totalRefunded).toFixed(2) ? 'REFUND' : 'PARTIALLY_REFUNDED';
  
      const refund = new Money((infoRefund.refunded_amount + totalRefunded).toFixed(2), order.getCurrencyCode());
  
      Transaction.wrap(function () {
        paymentInstrument.custom.addonPaymentStatus = paymentStatus;
        paymentInstrument.custom.addonPaymentRefund = refund.toString();
  
        if (paymentStatus === 'REFUND') {
          order.setPaymentStatus(Order.PAYMENT_STATUS_NOTPAID);
          order.setStatus(Order.ORDER_STATUS_CANCELLED);
        } else if (paymentStatus === 'PARTIALLY_REFUNDED') {
          order.setPaymentStatus(Order.PAYMENT_STATUS_PARTPAID);
        }
      });
  
    } else {
        var Logger = require('dw/system/Logger');
        Logger.error('Cannot find order with orderNo: {0}', orderNo);
    }
  }

module.exports = {
    calculateNonGiftCertificateAmount : calculateNonGiftCertificateAmount,
    createPaymentInstrument           : createPaymentInstrument,
    removeAddonPaymentInstrument      : removeAddonPaymentInstrument,
    getAddonPaymentInstrument         : getAddonPaymentInstrument,
    removeNonAddonPaymentInstrument   : removeNonAddonPaymentInstrument,
    checkTotalAmount                  : checkTotalAmount,
    updateQuixRefundTransaction       : updateQuixRefundTransaction
};
