'use strict';

const OrderMgr = require('dw/order/OrderMgr');
const Transaction = require('dw/system/Transaction');
const Order = require('dw/order/Order');
const Mac = require('dw/crypto/Mac');
const serviceName = 'int_addonPayments.http.rest';
const ServiceRegistry = require('dw/svc/LocalServiceRegistry');
const Resource = require('dw/web/Resource');

var Logger = require('dw/system/Logger');

const {
    createErrorMsg,
    getUrls
} = require('../addonPaymentsUtils');
const {
    merchantId,
    merchantKey,
    productId,
    merchantParams,
    operations,
    financing
} = require('../../../config/addonPaymentsPreferences');

const {
    calculateNonGiftCertificateAmount
} = require('./paymentInstrumentHelper');
const { createShippingAddress } = require('./addressHelper');

/**
 * Create token Authorization
 * @param {dw.order.Basket} currentBasket - user's basket
 * @returns {Object} with token unit data
 */
function getTokenAuthorization(currentBasket) {
    if (currentBasket) {
        let orderNo;

        Transaction.wrap(function () {
            orderNo =
        currentBasket instanceof Order
            ? currentBasket.orderNo
            : OrderMgr.createOrderNo();
        });
    }

    const authorizationInfo = {
        apiVersion      : '3',
        merchantId : merchantId,
        customerId : session.isCustomerAuthenticated()
            ? session.getCustomer().getProfile().getCustomerNo()
            : Math.floor(Math.random() * Date.now()),
        merchantKey : merchantKey,
        productId   : productId,
        country     : 'ES',
        // country     : currentBasket
        //     ? currentBasket.defaultShipment.shippingAddress.countryCode.getValue()
        //     : request.geolocation.countryCode,
        currency    : currentBasket
            ? currentBasket.currencyCode
            : session.getCurrency().getCurrencyCode(),
        operations         : operations,
        annonymousCustomer : !session.isCustomerAuthenticated()
    };

    return authorizationInfo;
}

/**
 * Validate Item
 * @param {Object} lineItem - line item
 * @returns {Object} with data
 */
function validateLineItem(lineItem) {
    let item = {
        article: {
            type                : 'product',
            name                : lineItem.lineItemText,
            category            : 'physical',
            unit_price_with_tax : lineItem.grossPrice.value,
            description         : lineItem.lineItemText
        },
        auto_shipping        : 'true',
        total_price_with_tax : lineItem.grossPrice.value
    };
    if (lineItem instanceof dw.order.ShippingLineItem) {
        item.article.category = 'shipping_fee';
        item.article.reference = lineItem.ID;
        item.units = 1;
    } else if (lineItem instanceof dw.order.PriceAdjustment) {
        item.article.category = 'discount';
        item.article.reference = lineItem.promotionID;
        item.units = lineItem.quantity;
    } else if (lineItem instanceof dw.order.ProductShippingLineItem) {
        item.article.category = 'shipping_fee';
        item.article.reference = lineItem.taxClassID;
        item.units = lineItem.quantity.value;
    } else {
        item.article.reference = lineItem.productID;
        item.units = lineItem.quantityValue;
    }

    return item;
}

/**
 * Get the attribute value or null
 * @param {*} attr - The attribute
 * @returns {* | null} - The value or null
 */
 function getValueOrNull(attr) {
    if (attr) {
        return attr;
    }
    return '';
}

/**
 * Creates payment
 * @param {dw.order.Basket} currentBasket - user's basket
 * @returns {Object} with payment data
 */

function getPaymentInfo(currentBasket, customerId, email, phone) {
    const dateHelpers = require('*/cartridge/scripts/helpers/dateHelpers');
    let orderNo;

    Transaction.wrap(function () {
        orderNo =
      currentBasket instanceof Order
          ? currentBasket.orderNo
          : OrderMgr.createOrderNo();
    });

    let customer = currentBasket.getCustomer() || undefined;
    let profile = customer.getProfile() || undefined;

    const paymentInfo = {
        merchantId      : merchantId,
        autoCapture     : !(merchantParams && merchantParams === 'AUTH'),
        customerId      : customerId,
        amount          :
      Math.round(calculateNonGiftCertificateAmount(currentBasket).value * 100) /
      100,
        country               : 'ES',
        languaje              : 'es',
        dob                   : profile ? dateHelpers.formatDate(profile.getBirthday(), 'yyyy/MM/dd') : '',
        merchantParams        : 'flow:' + merchantParams +
                                ';CMS_version:'+ Resource.msg('global.version.number', 'version', null) +
                                ';Plugin_Version:' + Resource.msg('addonPayments.version.number', 'addonPayments_version', null) + 
                                ';JS_version:' + Resource.msg('nodejs.version.number', 'addonPayments_version', null),
        //     country:
        //   currentBasket.defaultShipment.shippingAddress.countryCode.getValue(),
        currency              : currentBasket.currencyCode,
        statusURL             : getUrls().statusURL,
        errorURL              : getUrls().errorURL,
        successURL            : getUrls().successURL,
        awaitingURL           : getUrls().successURL,
        cancelURL             : getUrls().cancelURL,
        merchantTransactionId : (orderNo + '_' + Math.floor(Math.random() * Date.now())).slice(0, 12),
        operationType         : operations.toString() === 'deposit' ? 'debit' : 'credit',
        customerCountry       : 'es',
        customerEmail         : profile ? profile.email : email,
        productId   : Number(productId),
        chAddress1  : getValueOrNull(currentBasket.defaultShipment.shippingAddress.address1),
        chAddress2  : getValueOrNull(currentBasket.defaultShipment.shippingAddress.address2),
        chCity      : getValueOrNull(currentBasket.defaultShipment.shippingAddress.city),
        chPostCode  : getValueOrNull(currentBasket.defaultShipment.shippingAddress.postalCode),
        chCountry   : 'ES',
        // chCountry  : currentBasket.defaultShipment.shippingAddress.countryCode.getValue(),
        chEmail     : email,
        chFirstName : getValueOrNull(currentBasket.defaultShipment.shippingAddress.firstName),
        firstName   : getValueOrNull(currentBasket.defaultShipment.shippingAddress.firstName),
        chLastName  : getValueOrNull(currentBasket.defaultShipment.shippingAddress.lastName),
        lastName    : getValueOrNull(currentBasket.defaultShipment.shippingAddress.lastName),
        chPhone     : phone,
        chState     : getValueOrNull(currentBasket.defaultShipment.shippingAddress.stateCode)
    };
    const paysolExtendedData = {
        billing: {
            address: {
                city             : getValueOrNull(currentBasket.billingAddress.city),
                country          : 'ESP',
                postal_code      : getValueOrNull(currentBasket.billingAddress.postalCode),
                street_address   : getValueOrNull(currentBasket.billingAddress.address1),
                street_address_2 : getValueOrNull(currentBasket.billingAddress.address2)
            },
            company             : getValueOrNull(currentBasket.billingAddress.companyName),
            email               : paymentInfo.chEmail,
            first_name          : paymentInfo.chFirstName,
            last_name           : paymentInfo.chLastName,
            phone_number        : '%2B34' + paymentInfo.chPhone
        },
        cart: {
            currency             : currentBasket.currencyCode,
            reference            : currentBasket.UUID,
            total_price_with_tax : currentBasket.totalGrossPrice.value,
            items                : []
        },
        product  : financing,
        shipping : {
            address: {
                city             : paymentInfo.chCity,
                country          : 'ESP',
                postal_code      : paymentInfo.chPostCode,
                street_address   : paymentInfo.chAddress1,
                street_address_2 : paymentInfo.chAddress2
            },
            company      : '-',
            email        : paymentInfo.chEmail,
            first_name   : paymentInfo.chFirstName,
            last_name    : paymentInfo.chLastName,
            method       : 'home',
            name         : paymentInfo.chFirstName,
            phone_number : '%2B34' + paymentInfo.chPhone
        },
        confirmation_cart_data: { }
    };

    var allLineItems = currentBasket.allLineItems;
    var collections = require('*/cartridge/scripts/util/collections');
    // Add all products
    collections.forEach(allLineItems, function (lineItem) {
        let item = validateLineItem(lineItem);
        // Validate price is different to 0
        if (item.total_price_with_tax !== 0) {
            paysolExtendedData.cart.items.push(item);
        }
    });

    paymentInfo.paysolExtendedData = JSON.stringify(paysolExtendedData);
    // Add the type of financing 
    paymentInfo.financing = financing;

    return paymentInfo;
}

/**
 * Creates Validation 3DS V2
 * @param {Object} paymentInfo - Payment Info
 * @param {dw.order.Basket} currentBasket - user's basket
 */
function add3DSecure(paymentInfo, currentBasket) {
    paymentInfo.riskValidation3ds2 = addRiskValidationObject(currentBasket);
    paymentInfo.cdAccountInfo3ds2 = addCardHolderAccountObject(currentBasket);
    paymentInfo.shippingAddress = addShippingAddressObject(currentBasket);
}

/**
 * Creates Risk Validation 3DS V2
 * @param {dw.order.Basket} currentBasket - user's basket
 * @returns {Object} with risk validation data
 */
function addRiskValidationObject(currentBasket) {
    const codeDeliveryTime = currentBasket.defaultShipment.shippingMethodID;

    let deliveryTimeFrame = codeDeliveryTime.substring(
        codeDeliveryTime.length - 2,
        codeDeliveryTime.length
    );

    deliveryTimeFrame = deliveryTimeFrame === '03' ? deliveryTimeFrame : '04';

    const shippingAddress = currentBasket.defaultShipment.shippingAddress;

    let shippingMethod = '01';

    if (currentBasket.customer.registered) {
        currentBasket.customer.addressBook.addresses
            .toArray()
            .forEach(function (address) {
                if (address.isEquivalentAddress(shippingAddress)) {
                    shippingMethod = '02';
                }
            });
    }

    return {
        challenge         : '01',
        deliveryTimeFrame : deliveryTimeFrame,
        preOrderDate      : null,
        preOrderPurchase  : '01',
        reorderItems      : '01',
        shippingMethod    : shippingMethod,
        transactionTypeId : '01'
    };
}

/**
 * Creates Card Holder Account Information 3DS V2
 * @param {dw.order.Basket} currentBasket - user's basket
 * @returns {Object} with card holder account data
 */
function addCardHolderAccountObject(currentBasket) {
    const customer = session.getCustomer();
    const accountAge = getAccountAge(customer);

    let shippingAddressUsage = formatDate(new Date());

    if (customer.registered) {
        customer.addressBook.addresses.toArray().forEach(function (address) {
            if (
                address.isEquivalentAddress(
                    currentBasket.defaultShipment.shippingAddress
                )
            ) {
                shippingAddressUsage = formatDate(address.creationDate);
            }
        });
    }

    return {
        accountAge          : accountAge,
        accountChangeDate   : null,
        accountCreationDate : customer.registered
            ? formatDate(customer.profile.creationDate)
            : null,
        passwordChangeDate      : null,
        paymentAccountAge       : null,
        provisioningAttemptsDay : null,
        purchasesLast6Months    : customer.registered
            ? getOrders(customer, 6, 'm')
            : null,
        shippingAddressUsage         : shippingAddressUsage,
        shippingNameInd              : '01',
        suspiciousAccountActivityInd : '02',
        txnActivityDay               : customer.registered ? getOrders(customer, 1, 'd') : null,
        txnActivityYear              : customer.registered ? getOrders(customer, 1, 'y') : null
    };
}

/**
 * Creates Account Age for Card Holder Account Information
 * @param {dw.customer.Customer} customer - customer
 * @returns {Object} with account age data
 */
function getAccountAge(customer) {
    let accountAge;
    if (customer.registered) {
        const creationDateCustomer = customer.profile.creationDate.getTime();
        const today = new Date().getTime();

        const difference = Math.abs(today - creationDateCustomer);
        const days = Math.ceil(difference / (1000 * 3600 * 24));

        if (days <= 1) {
            accountAge = '02';
        } else if (days > 1 && days < 30) {
            accountAge = '03';
        } else if (days >= 30 && days < 60) {
            accountAge = '04';
        } else {
            accountAge = '05';
        }
    } else {
        accountAge = '01';
    }

    return accountAge;
}

/**
 * Format Date
 * @param { Date} date - date
 * @returns {Object} with format date
 */
function formatDate(date) {
    const month =
    date.getMonth().toString().length === 1
        ? '0' + date.getMonth()
        : date.getMonth();

    const formatDate =
    date.getDate().toString().length === 1
        ? '0' + date.getDate()
        : date.getDate();

    return date.getFullYear() + '-' + month + '-' + formatDate;
}

/**
 * Return number of orders in period of time
 * @param {dw.customer.Customer} customer - customer
 * @param {Number} time - Time
 * @param {String} method - Day, month or year
 * @returns {Object} with number of orders
 */
function getOrders(customer, time, method) {
    let today = new Date();
    switch (method) {
    case 'd':
        today.setDate(today.getDate() - time);
        break;
    case 'm':
        today.setMonth(today.getMonth() - time);
        break;

    case 'y':
        today.setFullYear(today.getFullYear() - time);
        break;
    }

    const orders = customer
        .getOrderHistory()
        .getOrders('creationDate > {0}', null, today);

    orders.close();

    return orders.count;
}

/**
 * Creates Shipping Address 3DS V2
 * @param {dw.order.Basket} currentBasket - user's basket
 * @returns {Object} with shipping address data
 */
function addShippingAddressObject(currentBasket) {
    return {
        address1           : currentBasket.billingAddress.address1,
        address2           : currentBasket.billingAddress.address2,
        city               : currentBasket.billingAddress.city,
        zipCode            : currentBasket.billingAddress.postalCode,
        country            : currentBasket.billingAddress.stateCode,
        phone              : currentBasket.billingAddress.phone,
        deliveryEmail      : currentBasket.customerEmail,
        billAddressMatchYN : true
    };
}

/**
 * Create purchase unit description based on items in the basket
 * @param  {dw.order.ProductLineItem} productLineItems Items in the basket
 * @returns {string} item description
 */
function getItemsDescription(productLineItems) {
    return Array.map(productLineItems, function (productLineItem) {
        return productLineItem.productName;
    })
        .join(',')
        .substring(0, 127);
}

/**
 * @param  {dw.value.Money} acc current basket order + product discount
 * @param  {dw.order.OrderPaymentInstrument} giftCertificate GC from the basket
 * @returns {dw.value.Money} Gift certificate cotal
 */
function getAppliedGiftCertificateTotal(acc, giftCertificate) {
    return acc.add(giftCertificate.paymentTransaction.amount);
}

/**
 * Creates puchase unit data
 * @param {dw.order.Basket} currentBasket - user's basket
 * @param {boolean} isCartFlow - whether from cart or no
 * @returns {Object} with purchase unit data
 */
function getPurchaseUnit(currentBasket, isCartFlow) {
    const {
        currencyCode,
        defaultShipment,
        productLineItems,
        totalTax,
        shippingTotalPrice,
        adjustedShippingTotalPrice,
        merchandizeTotalPrice,
        adjustedMerchandizeTotalPrice,
        giftCertificateTotalPrice
    } = currentBasket;
    let orderNo;
    let handling;
    let insurance;

    Transaction.wrap(function () {
        orderNo =
      currentBasket instanceof Order
          ? currentBasket.orderNo
          : OrderMgr.createOrderNo();
    });
    const nonShippingDiscount = Array.reduce(
        currentBasket.giftCertificatePaymentInstruments,
        getAppliedGiftCertificateTotal,
        merchandizeTotalPrice.subtract(adjustedMerchandizeTotalPrice)
    );

    const purchaseUnit = {
        description : getItemsDescription(productLineItems),
        amount      : {
            currency_code : currencyCode,
            value         : calculateNonGiftCertificateAmount(currentBasket).value.toString(),
            breakdown     : {
                item_total: {
                    currency_code : currencyCode,
                    value         : merchandizeTotalPrice
                        .add(giftCertificateTotalPrice)
                        .value.toString()
                },
                shipping: {
                    currency_code : currencyCode,
                    value         : shippingTotalPrice.value.toString()
                },
                tax_total: {
                    currency_code : currencyCode,
                    value         : totalTax.value.toString()
                },
                handling: {
                    currency_code : currencyCode,
                    value         : !empty(handling) ? handling : '0'
                },
                insurance: {
                    currency_code : currencyCode,
                    value         : !empty(insurance) ? insurance : '0'
                },
                shipping_discount: {
                    currency_code : currencyCode,
                    value         : shippingTotalPrice
                        .subtract(adjustedShippingTotalPrice)
                        .value.toString()
                },
                discount: {
                    currency_code : currencyCode,
                    value         : nonShippingDiscount.value.toString()
                }
            }
        },
        invoice_id: orderNo
    };
    if (!isCartFlow && defaultShipment && defaultShipment.getShippingAddress()) {
        purchaseUnit.shipping = createShippingAddress(
            defaultShipment.getShippingAddress()
        );
    }
    return purchaseUnit;
}

function toHexString(byteArray) {
    var s = '';
    for (var i = 0; i < byteArray.getLength(); i++) {
        s += ('0' + (byteArray.byteAt(i) & 0xff).toString(16)).slice(-2);
    }
    return s;
}

function verifySignature(body, signature) {
    var restService = ServiceRegistry.createService(serviceName, {});
    var signatureSplited = signature.split(',');
    var params = {};
    for (var n = 0; n < signatureSplited.length; n++) {
        var part = signatureSplited[n];
        var [key, value] = part.split('=');
        params[key] = value;
    }
    var hmac = Mac(Mac.HMAC_SHA_256);
    var result = hmac.digest(
        params.t + '.' + body,
        restService.configuration.credential.password
    );

    if (toHexString(result) !== params.v1) {
        throw new Error(createErrorMsg('signature'));
    }

    return JSON.parse(body);
}

/**
 * Get the order
 * @param {string} orderNo - The order number
 * @param {Object} orderToken - The order token
 * @returns {dw.order.Order | null} - The order or null
 */
 function getOrder(orderNo, orderToken) {
    var OrderMgr = require('dw/order/OrderMgr');
    if (!orderNo || !orderToken) {
        return null;
    }

    return OrderMgr.getOrder(orderNo, orderToken);

}

/**
 * Stores the payment id in session
 * @param {Object} paymentResponse - The paymentResponse
 */
function setPaymentIdInSession(paymentResponse) {
    let addonPaymentId;
    if(paymentResponse.err) return;
    if(paymentResponse.response && paymentResponse.response.operationSize > 0){
        addonPaymentId = paymentResponse.response.operationsArray[0].payFrexTransactionId.toString();
    } else {
        const xmlStatus = new XML(paymentResponse.response);
        const xmlValues = xmlStatus.elements('operations').elements('operation');
        const size = xmlValues.length();
        addonPaymentId = xmlValues[size - 1].payFrexTransactionId.toString();
    }

    session.custom.addonPaymentId = addonPaymentId;

}

/**
 * Deletes the payment id in session
 */
 function deletePaymentIdInSession() {
    session.custom.addonPaymentId = null;
}

/**
 * Set the payment status not paid
 * @param {dw.order.Order} order - The order
 * @returns {boolean} - Return true if the order status has been updated
 */
 function setPaymentStatusNotPaid(order) {
    var Order = require('dw/order/Order');
    var OrderMgr = require('dw/order/OrderMgr');
    var Transaction = require('dw/system/Transaction');

    if (!order) {
        return false;
    }

    try {
        Transaction.wrap(function () {
            order.setPaymentStatus(Order.PAYMENT_STATUS_NOTPAID);
            OrderMgr.failOrder(order, true);
        });
        return true;
    } catch (e) {
        Logger.error('Cannot set the payment status not paid or fail the order. OrderNo: {0} - Error {1}', order.orderNo, e);
        return false;
    }
}

/**
 * Encode the order token
 * @param {Object | null} order - The order
 */
 function setOrderTokenInSession(order) {
    var cryptography = require('*/cartridge/scripts/addonPayments/cryptography');

    if (order) {        
        var orderTo = cryptography.encodeMerchantParameters(order.orderToken);
        if (orderTo) {
            session.custom.orderTo = orderTo;
        }
    }

}

/**
 * Decode the order token
 * @returns {Object | null} - The decoded response or null
 */
 function getOrderTokenInSession() {
    var cryptography = require('*/cartridge/scripts/addonPayments/cryptography');
    const order = session.custom.orderTo;
    if (!order) {
        return null;
    }

    return cryptography.decodeMerchantParameters(order);
}

/**
 * Update the order with payment info
 * @param {dw.order.Order} order - The order
 * @returns {Object} - The result
 */
 function orderCompleteFields(order) {
    var Order = require('dw/order/Order');
    var Locale = require('dw/util/Locale');
    var CustomerMgr = require('dw/customer/CustomerMgr');
    var OrderMgr = require('dw/order/OrderMgr');
    var Transaction = require('dw/system/Transaction');
    var checkoutHelpers = require('*/cartridge/scripts/checkout/checkoutHelpers');
    var customValidationHelpers = require('*/cartridge/scripts/helpers/customValidationHelpers');

    var result = {};

    if (customValidationHelpers.checkDwCustomType(order, 'dw.order.Order', dw.order.Order, true)) {
        checkoutHelpers.saveAddressCustomer(order);

        try {
            Transaction.wrap(function () {
                
                OrderMgr.placeOrder(order);
                var customer = CustomerMgr.getCustomerByLogin(order.customerEmail);
                if (customer && !order.customerNo && customer.profile) {
                    order.setCustomerNo(customer.profile.customerNo);
                    order.setCustomer(customer);
                }
                order.setPaymentStatus(Order.PAYMENT_STATUS_PAID);
                order.setStatus(Order.ORDER_STATUS_OPEN);
                order.setExportStatus(Order.EXPORT_STATUS_READY);
                order.setConfirmationStatus(Order.CONFIRMATION_STATUS_CONFIRMED);

            });

            if (order.getCustomerEmail()) {
                var locale = Locale.getLocale(request.getLocale());
                checkoutHelpers.sendConfirmationEmail(order, locale.getID());
            }
        } catch (e) {
            Logger.error('Cannot update the payment status. OrderNo {0}: ', order.orderNo);
            result.error = true;
            return result;
        }

        // Order custom attributes
        result = checkoutHelpers.saveCustomAttributes(order);

        result.orderNo = order.orderNo;
        result.orderToken = order.orderToken;
    } else {
        result.error = true;
    }

    return result;
}

module.exports = {
    getPurchaseUnit          : getPurchaseUnit,
    verifySignature          : verifySignature,
    getTokenAuthorization    : getTokenAuthorization,
    getPaymentInfo           : getPaymentInfo,
    add3DSecure              : add3DSecure,
    getOrders                : getOrders,
    getOrder                 : getOrder,
    setPaymentIdInSession    : setPaymentIdInSession,
    deletePaymentIdInSession : deletePaymentIdInSession,
    setPaymentStatusNotPaid  : setPaymentStatusNotPaid,
    setOrderTokenInSession   : setOrderTokenInSession,
    getOrderTokenInSession   : getOrderTokenInSession,
    orderCompleteFields      : orderCompleteFields
};
