'use strict';

const Encoding = require('dw/crypto/Encoding');
const Bytes = require('dw/util/Bytes');
const URLUtils = require('dw/web/URLUtils');
const Resource = require('dw/web/Resource');
const Transaction = require('dw/system/Transaction');
const { queryCustomObjects, remove } = require('dw/object/CustomObjectMgr');
const {
    successUrl,
    cancelUrl,
    errorUrl
} = require('../../config/addonPaymentsPreferences');

let addonPaymentsLogger;

/**
 * Encodes purchase unit object into encoded string
 *
 * @param {Object} purchaseUnit purchase unit
 * @returns {string} encoded string
 */
function encodeString(purchaseUnit) {
    const bytes = new Bytes(JSON.stringify(purchaseUnit));
    return Encoding.toBase64(bytes);
}

/**
 * Get logger instance
 *
 * @param {string} err Error message
 */
function createErrorLog(err) {
    var Logger = require('dw/system/Logger');
    addonPaymentsLogger =
    addonPaymentsLogger ||
    Logger.getLogger('AddonPayments', 'Addon_Payments_General');
    if (!empty(err)) {
        addonPaymentsLogger.error(err.stack ? err.message + err.stack : err);
    } else {
        addonPaymentsLogger.debug('Empty log entry');
    }
    return;
}

/**
 * Create URL for a call
 * @param  {string} host
 * @param  {string} path REST action endpoint
 * @returns {string} url for a call
 */
function getUrlPath(host, path) {
    var url = host;
    if (!url.match(/.+\/$/)) {
        if (!path.match(/^\//)) {
            url += '/';
        }
    } else if (path.match(/^\//)) {
        return (url += path.substring(1));
    }
    url += path;
    return url;
}

function getUrls() {
    return {
        paymentStage: URLUtils.https(
            'Checkout-Begin',
            'stage',
            'payment'
        ).toString(),
        placeOrderStage  : URLUtils.https('AddonPayments-OnlineResponse').toString(),
        getAuthorization : URLUtils.https('AddonPayments-Authorization').toString(),
        callbackUrl      : URLUtils.https('AddonPayments-Payments').toString(),
        createOrderURL   : URLUtils.https('AddonPayments-CreateOrder').toString(),
        errorURL         : URLUtils.https(errorUrl).toString(),
        cancelURL        : URLUtils.https(cancelUrl).toString(),
        successURL       : URLUtils.https(successUrl).toString(),
        statusURL        : URLUtils.https('AddonPayments-Status').toString()
    };
}

function getAddonPaymentsNewTransaction(orderId, addonPaymentId) {
    const newTransaction = queryCustomObjects(
        'AddonPaymentsNewTransactions',
        'custom.orderNo = {0} and custom.addonPaymentId = {1}',
        null,
        orderId,
        addonPaymentId
    );

    return newTransaction;
}

function deleteAddonPaymentsNewTransaction(orderId, addonPaymentId) {
    Transaction.wrap(function () {
        const transactionToRemove = queryCustomObjects(
            'AddonPaymentsNewTransactions',
            'custom.orderNo = {0} and custom.addonPaymentId = {1}',
            null,
            orderId,
            addonPaymentId
        );
        while (transactionToRemove.hasNext()) {
            remove(transactionToRemove.next());
        }
    });
}

/**
 * Creates the Error Message
 *
 * @param {string} errorName error message name
 * @returns {string} errorMsg - Resource error massage
 */
function createErrorMsg(errorName) {
    const errorMsg = Resource.msg(
        'addonPayments.error.' + errorName,
        'addonPaymentsErrors',
        errorName
    );
    return errorMsg;
}

module.exports = {
    createErrorLog                    : createErrorLog,
    encodeString                      : encodeString,
    createErrorMsg                    : createErrorMsg,
    getUrlPath                        : getUrlPath,
    getUrls                           : getUrls,
    getAddonPaymentsNewTransaction    : getAddonPaymentsNewTransaction,
    deleteAddonPaymentsNewTransaction : deleteAddonPaymentsNewTransaction
};
