'use strict';

const page = module.superModule;
const server = require('server');

const csrfProtection = require('*/cartridge/scripts/middleware/csrf');
const userLoggedIn = require('*/cartridge/scripts/middleware/userLoggedIn');
const consentTracking = require('*/cartridge/scripts/middleware/consentTracking');

server.extend(page);

/**
 * Creates an object from form values
 * @param {Object} paymentForm - form object
 * @returns {Object} a plain object of payment instrument
 */
function getDetailsObject(paymentForm) {
    let name;
    let cardNumber;
    let cardType;
    let expirationMonth;
    let expirationYear;

    paymentForm.forEach(function (element) {
        switch (element.name) {
        case 'cardType':
            cardType = element.value;
            break;
        case 'cardNumber':
            cardNumber = element.value;
            break;
        case 'chName':
            name = element.value;
            break;
        case 'expDate':
            expirationMonth = element.value.substring(0, element.value.length - 2);
            expirationYear = element.value.substring(
                element.value.length - 2,
                element.value.length
            );
            break;
        default:
        }
    });

    return {
        name            : name,
        cardNumber      : cardNumber,
        cardType        : cardType,
        expirationMonth : Number(expirationMonth),
        expirationYear  : Number(expirationYear)
    };
}

server.append(
    'List',
    userLoggedIn.validateLoggedIn,
    consentTracking.consent,
    function (req, res, next) {
        const prefs = require('../config/addonPaymentsPreferences');
        res.render('account/payment/payment', {
            prefs: prefs
        });
        next();
    }
);

server.append(
    'AddPayment',
    csrfProtection.generateToken,
    consentTracking.consent,
    userLoggedIn.validateLoggedIn,
    function (req, res, next) {
        const prefs = require('../config/addonPaymentsPreferences');
        res.render('account/payment/addPayment', {
            prefs: prefs
        });

        next();
    }
);

server.post(
    'SaveAddonPayments',
    userLoggedIn.validateLoggedIn,
    function (req, res, next) {
        const accountHelpers = require('*/cartridge/scripts/helpers/accountHelpers');

        const paymentForm = JSON.parse(req.body);
        const result = getDetailsObject(paymentForm);

        res.setViewData(result);
        this.on('route:BeforeComplete', function (req, res) {
            // eslint-disable-line no-shadow
            const URLUtils = require('dw/web/URLUtils');
            const CustomerMgr = require('dw/customer/CustomerMgr');
            const Transaction = require('dw/system/Transaction');

            const formInfo = res.getViewData();
            const customer = CustomerMgr.getCustomerByCustomerNumber(
                req.currentCustomer.profile.customerNo
            );
            const wallet = customer.getProfile().getWallet();

            Transaction.wrap(function () {
                const paymentInstrument =
          wallet.createPaymentInstrument('ADDON_PAYMENTS');
                paymentInstrument.setCreditCardHolder(formInfo.name);
                paymentInstrument.setCreditCardNumber(formInfo.cardNumber);
                paymentInstrument.setCreditCardType(formInfo.cardType);
                paymentInstrument.setCreditCardExpirationMonth(
                    formInfo.expirationMonth
                );
                paymentInstrument.setCreditCardExpirationYear(formInfo.expirationYear);

                const token = formInfo.cardNumber;

                paymentInstrument.setCreditCardToken(token);
                paymentInstrument.custom.transactionsHistory = 0;
            });

            // Send account edited email
            accountHelpers.sendAccountEditedEmail(customer.profile);

            res.json({
                success     : true,
                redirectUrl : URLUtils.url('PaymentInstruments-List').toString()
            });
        });

        return next();
    }
);

module.exports = server.exports();
