'use strict';

var server = require('server');
server.extend(module.superModule);

var Resource = require('dw/web/Resource');
var URLUtils = require('dw/web/URLUtils');

var csrfProtection = require('*/cartridge/scripts/middleware/csrf');
var consentTracking = require('*/cartridge/scripts/middleware/consentTracking');

var OrderMgr = require('dw/order/OrderMgr');

const {
    getAddonPaymentInstrument
} = require('../scripts/addonPayments/helpers/paymentInstrumentHelper');

server.append('Confirm', function (req, res, next) {
    const order = OrderMgr.getOrder(req.form.orderID);
    const addonPaymentInstrument = getAddonPaymentInstrument(order);

    if (!addonPaymentInstrument) {
        next();
        return;
    }

    const paymentSolution = addonPaymentInstrument.custom.addonPaymentSolution;

    res.setViewData({
        addonPayments: {
            paymentSolution: paymentSolution
        }
    });
    next();
});

server.append('Details', function (req, res, next) {
    const order = OrderMgr.getOrder(req.querystring.orderID);
    const addonPaymentInstrument = getAddonPaymentInstrument(order);

    if (!addonPaymentInstrument) {
        next();
        return;
    }

    const paymentSolution = addonPaymentInstrument.custom.addonPaymentSolution;

    res.setViewData({
        addonPayments: {
            paymentSolution : paymentSolution
        }
    });
    next();
});

/**
 * Order-GetAddonPaymentsConfirm: This endpoint is invoked when the shopper's Order is Placed and Confirmed
 * @name Base/Order-GetAddonPaymentsConfirm
 * @function
 * @memberof Order
 */
 server.get(
    'GetAddonPaymentsConfirm',
    consentTracking.consent,
    server.middleware.https,
    csrfProtection.generateToken,
    function (req, res, next) {
        var reportingUrlsHelper = require('*/cartridge/scripts/reportingUrls');
        var OrderMgr = require('dw/order/OrderMgr');
        var OrderModel = require('*/cartridge/models/order');
        var Locale = require('dw/util/Locale');

        var orderNo = req.querystring.orderNo;
        var orderToken = req.querystring.orderToken;

        if (!orderNo || !orderToken) {
            res.render('/error', {
                message: Resource.msg('error.confirmation.error', 'confirmation', null)
            });

            return next();
        }

        var order = OrderMgr.getOrder(String(orderNo), String(orderToken));

        if (!order || order.customer.ID !== req.currentCustomer.raw.ID
        ) {
            res.render('/error', {
                message: Resource.msg('error.confirmation.error', 'confirmation', null)
            });

            return next();
        }
        var lastOrderID = Object.prototype.hasOwnProperty.call(req.session.raw.custom, 'orderID') ? req.session.raw.custom.orderID : null;
        if (lastOrderID === req.querystring.ID) {
            res.redirect(URLUtils.url('Home-Show'));
            return next();
        }

        var config = {
            numberOfLineItems: '*'
        };

        var currentLocale = Locale.getLocale(req.locale.id);

        var orderModel = new OrderModel(
            order,
            { config: config, countryCode: currentLocale.country, containerView: 'order' }
        );
        var passwordForm;

        var reportingURLs = reportingUrlsHelper.getOrderReportingURLs(order);
        var addonPaymentInstrument = getAddonPaymentInstrument(order);
        var addonPayments = {
            paymentSolution : addonPaymentInstrument && addonPaymentInstrument.custom ? addonPaymentInstrument.custom.addonPaymentSolution : ''
        }

        if (!req.currentCustomer.profile) {
            passwordForm = server.forms.getForm('newPasswords');
            passwordForm.clear();
            res.render('checkout/confirmation/confirmation', {
                order             : orderModel,
                returningCustomer : false,
                passwordForm      : passwordForm,
                reportingURLs     : reportingURLs,
                addonPayments     : addonPayments,
                orderUUID         : order.getUUID()
            });
        } else {
            res.render('checkout/confirmation/confirmation', {
                order             : orderModel,
                returningCustomer : true,
                reportingURLs     : reportingURLs,
                addonPayments     : addonPayments,
                orderUUID         : order.getUUID()
            });
        }

        req.session.raw.custom.orderID = orderNo;

        return next();
    }
);

/**
 * Order-History : This endpoint is invoked to get Order History for the logged in shopper
 * @name Base/Order-History
 * @function
 * @memberof Order
 */
 server.append('History', function (req, res, next) {
        var viewData = res.getViewData();
        if (viewData) {
            // Filter ordes
            var orders = viewData.orders.filter(function (order)  { return order.orderStatus.value !== 0 && order.orderStatus.value !== 8 });
            viewData.orders = orders;
        }

        next();
    }
);


module.exports = server.exports();
