'use strict';

const server = require('server');
const Transaction = require('dw/system/Transaction');
const HookMgr = require('dw/system/HookMgr');
const Status = require('dw/system/Status');
const Resource = require('dw/web/Resource');
const URLUtils = require('dw/web/URLUtils');
const PaymentMgr = require('dw/order/PaymentMgr');
const OrderMgr = require('dw/order/OrderMgr');
const Order = require('dw/order/Order');
const Locale = require('dw/util/Locale');
const { createCustomObject } = require('dw/object/CustomObjectMgr');
const Logger = require('dw/system/Logger');
const checkoutHelpers = require('*/cartridge/scripts/checkout/checkoutHelpers');
const addonPaymentHelpers = require('../scripts/addonPayments/helpers/addonPaymentsHelper');
const paymentInstrumentHelper = require('../scripts/addonPayments/helpers/paymentInstrumentHelper');

const {
    createToken,
    createPayment
} = require('../scripts/addonPayments/addonPaymentsAPI');

const {
    getAddonPaymentsNewTransaction,
    encodeString,
    createErrorMsg,
    createErrorLog,
    getUrls
} = require('../scripts/addonPayments/addonPaymentsUtils');

const {
    createPaymentInstrument,
    getAddonPaymentInstrument,
    removeAddonPaymentInstrument,
    removeNonAddonPaymentInstrument,
    checkTotalAmount
} = require('../scripts/addonPayments/helpers/paymentInstrumentHelper');

const {
    updateOrderBillingAddress
} = require('../scripts/addonPayments/helpers/addressHelper');

server.get('CreateToken', server.middleware.https, function (req, res, next) {
    const { currentBasket } = require('dw/order/BasketMgr');
    
    // Generate token
    const token = addonPaymentHelpers.getTokenAuthorization(currentBasket);
    const tokenResponse = createToken(token);

    res.json({
        tokenAuthorization: tokenResponse
    });

    next();
});

/**
 * AddonPayments-Authorization: The AddonPayments create authorization
 * @name AddonPayments-Authorization
 * @function
 * @memberof AddonPayments
 */
server.post(
    'Authorization',
    server.middleware.https,
    function (req, res, next) {
        const { currentBasket } = require('dw/order/BasketMgr');
        const details = JSON.parse(req.body);

        // CHECK MAIL AND PHONE
        if (details) {
            const formFieldErrors = {};
            let message;
            const regex = /^[\w.%+-]+@[\w.-]+\.[\w]{2,6}$/;
            if (!details.email || details.email.trim().length === 0) {
                message = Resource.msg('error.card.info.missing.email', 'forms', null);
                formFieldErrors.emailError = message;
            } else if (!regex.test(details.email)) {
                message = Resource.msg(
                    'error.message.parse.email.profile.form',
                    'forms',
                    null
                );
                formFieldErrors.emailError = message;
            }
            if (!details.phone || details.phone.trim().length === 0) {
                message = Resource.msg('error.card.info.missing.phone', 'forms', null);
                formFieldErrors.phoneError = message;
            } else if (details.phone.trim().length !== 9) {
                message = Resource.msg('error.card.info.phone.invalid', 'forms', null);
                formFieldErrors.phoneError = message;
            }
        } else {
            message = Resource.msg('error.card.info.missing.email', 'forms', null);
            formFieldErrors.emailError = message;
            message = Resource.msg('error.card.info.missing.phone', 'forms', null);
            formFieldErrors.phoneError = message;
        }

        if (formFieldErrors.emailError || formFieldErrors.phoneError) {
            res.json({
                fieldErrors : formFieldErrors,
                error       : true
            });
            return next();
        }

        // Generate token
        const token = addonPaymentHelpers.getTokenAuthorization(currentBasket);
        session.privacy.paymentEmail = details.email;
        session.privacy.paymentPhone = details.phone;
        const currentLocale = Locale.getLocale(req.locale.id);

        const tokenResponse = createToken(token);
        const payment = addonPaymentHelpers.getPaymentInfo(
            currentBasket,
            token.customerId,
            details.email,
            details.phone
        );
        payment.ipAddress = req.remoteAddress;

        // Update billing address.
        updateOrderBillingAddress(currentBasket, payment);

        // Return token
        res.json({
            tokenAuthorization : tokenResponse,
            payment            : payment,
            callbackUrl        : getUrls().callbackUrl,
            language           : currentLocale.language
        });

        next();
    }
);


/**
 * AddonPayments-Payments: The AddonPayments create payment
 * @name AddonPayments-Payments
 * @function
 * @memberof AddonPayments
 */
server.post('Payments', server.middleware.https, function (req, res, next) {
    const { currentBasket } = require('dw/order/BasketMgr');
    const payment = JSON.parse(req.body);
    if (checkTotalAmount(currentBasket, payment.amount)) {
        const token = req.httpHeaders.get('prepaytoken').replace(/['"]+/g, '');
        addonPaymentHelpers.add3DSecure(payment, currentBasket);
        const paymentResponse = createPayment(payment, token);
        // Save the addon payment id in session
        addonPaymentHelpers.setPaymentIdInSession(paymentResponse);
        if (paymentResponse.err || (paymentResponse.response && paymentResponse.response.status !== 'ERROR')) {
            // tenemos el payment lo mandamos a través del API
            res.json({
                paymentResponse : paymentResponse,
                createOrderUrl  : getUrls().createOrderURL
            });
        } else {
            res.json({
                paymentResponse: { err: createErrorMsg(paymentResponse.err || paymentResponse.response.message) }
            });
        }
    } else {
        res.json({
            paymentResponse: { err: createErrorMsg('expiredpayment') }
        });
    }
    next();
});

/**
 * AddonPayments-CreateOrder: The AddonPayments create order
 * @name AddonPayments-CreateOrder
 * @function
 * @memberof AddonPayments
 */
server.post('CreateOrder', server.middleware.https, function (req, res, next) {
    const Money = require('dw/value/Money');
    const CustomerMgr = require('dw/customer/CustomerMgr');

    const { currentBasket } = require('dw/order/BasketMgr');
    
    if (!currentBasket) {
        var failOrderBasketId = checkoutHelpers.getOrderNoInSession();
        if (failOrderBasketId) {
            res.redirect(URLUtils.url('Checkout-Begin', 'stage', 'payment'));
        } else {
            res.redirect(URLUtils.url('Cart-Show').toString());
        }
        return next();
    }

    const createOrderResult = checkoutHelpers.createNewOrder(currentBasket);
    if (createOrderResult.error || !createOrderResult.order) {
        if ('redirectUrl' in createOrderResult) {
            res.redirect(createOrderResult.redirectUrl);
        } else {
            res.redirect(URLUtils.url('Cart-Show').toString());
        }

        if ('errorMessage' in createOrderResult) {
            Logger.warn('AddonPayments - Cannot create the order: {0}', createOrderResult.errorMessage);
        }

        return next();
    }
    
    const order = createOrderResult.order;
    const orderNo = order.orderNo;

    // Save the order number in session
    checkoutHelpers.setOrderNoInSession(order);
    addonPaymentHelpers.setOrderTokenInSession(order);
    
    const body = req.body && JSON.parse(req.body);
    const paymentRefund = new Money(0, currentBasket.getCurrencyCode());
    const paymentInstrument = createPaymentInstrument(
        order,
        'ADDON_PAYMENTS'
    );
    const paymentProcessor =
    PaymentMgr.getPaymentMethod('ADDON_PAYMENTS').getPaymentProcessor();
    Transaction.wrap(function () {
        paymentInstrument.paymentTransaction.setPaymentProcessor(paymentProcessor);
        paymentInstrument.custom.addonPaymentID = body.addonPaymentID;
        paymentInstrument.custom.addonPaymentsMerchantID =
        body.merchantTransactionId;
        paymentInstrument.custom.addonPaymentSolution =
        body.paymentSolution.toUpperCase();
        paymentInstrument.custom.addonPaymentStatus = body.status;
        paymentInstrument.custom.addonPaymentRefund = paymentRefund.toString();
    });

    

    if (body.paymentSolution !== 'bizum' && (body.status === 'SUCCESS' || body.status === 'PENDING' || body.status === 'REDIRECTED')) {
        const addonPaymentsTransaction = getAddonPaymentsNewTransaction(
            orderNo,
            body.addonPaymentID.toString()
        );

        if (addonPaymentsTransaction && !addonPaymentsTransaction.hasNext()) {
            Transaction.wrap(function () {
                const newTransaction = createCustomObject(
                    'AddonPaymentsNewTransactions',
                    orderNo
                );
                newTransaction.custom.addonPaymentId = body.addonPaymentID;
                newTransaction.custom.addonPaymentStatus = body.status;
            });
        }
    }

    const customerNo = currentBasket.getCustomerNo();

    if (customerNo) {
        const customer = CustomerMgr.getCustomerByCustomerNumber(customerNo);
        const wallet = customer.getProfile().getWallet();

        const payment = wallet.getPaymentInstruments('ADDON_PAYMENTS');

        payment.toArray().forEach(function (addonPayment) {
            if (body.paymentToken === addonPayment.getCreditCardToken()) {
                Transaction.wrap(function () {
                    addonPayment.custom.transactionsHistory += 1;
                });
            }
        });
    }
    

    return;
});

/**
 * AddonPayments-Status: The AddonPayments async response
 * @name AddonPayments-Status
 * @function
 * @memberof AddonPayments
 */
server.post('Status', server.middleware.https, function (req, res, next) {
    if (req.body) {
        const xmlStatus = new XML(req.body);
        const xmlValues = xmlStatus.elements('operations').elements('operation');
        const size = xmlValues.length();
        const status = xmlValues[size - 1].status.toString();
        const paymentSolution = xmlValues[size - 1].paymentSolution.toString();
        const details = xmlValues[size - 1].details.toString();
        const operationType = xmlValues[size - 1].operationType.toString();
        const originalAmount = xmlValues[size - 1].originalAmount.toString();
        const xmlToObject = xmlStatus.toString();
        

        if ((status === 'SUCCESS' || status === 'PENDING') && operationType !== 'REBATE') {
            let addonPaymentId = xmlValues[size - 1].payFrexTransactionId.toString();

            let orderByMerchant = xmlValues[size - 1].merchantTransactionId
                .toString()
                .split('_')[0];

            // Change status paymentInstrument .
            let addonPaymentsTransaction = getAddonPaymentsNewTransaction(
                orderByMerchant,
                addonPaymentId
            );

            if (addonPaymentsTransaction && !addonPaymentsTransaction.hasNext()) {
                Transaction.wrap(function () {
                    const newTransaction = createCustomObject(
                        'AddonPaymentsNewTransactions',
                        orderByMerchant
                    );
                    newTransaction.custom.addonPaymentId = addonPaymentId;
                    newTransaction.custom.addonPaymentStatus = status;
                });
            }
        }
        if (paymentSolution === 'quix' && (status === 'SUCCESS' || status === 'CANCELLED') && operationType !== 'REBATE') {
            let addonPaymentId = xmlValues[size - 1].payFrexTransactionId.toString();

            let orderByMerchant = xmlValues[size - 1].merchantTransactionId
                .toString()
                .split('_')[0];

            // Change status paymentInstrument .
            let addonPaymentsTransaction = getAddonPaymentsNewTransaction(
                orderByMerchant,
                addonPaymentId
            );
            if (addonPaymentsTransaction && addonPaymentsTransaction.hasNext()) {
                Transaction.wrap(function () {
                    const addonPaymentsOrdersId = addonPaymentsTransaction.next();
                    addonPaymentsOrdersId.custom.addonPaymentStatus = status;
                });
            }
        }
        if (paymentSolution === 'quix' && status === 'SUCCESS' && operationType === 'REBATE'){

            var xmlOptionalParams = xmlStatus.elements('optionalTransactionParams').elements('entry');
            const sizeOptions = xmlOptionalParams.length();
            const responseKey = xmlOptionalParams[sizeOptions - 1].key.toString();
            const responseOrderToken = xmlOptionalParams[sizeOptions - 1].value.toString();

            if (responseKey && responseKey === 'orderToken' && responseOrderToken){
                var infoRefund = JSON.parse(details);
                var orderNo = infoRefund.order_reference_1.replace(/_[\d]+/, "");
                paymentInstrumentHelper.updateQuixRefundTransaction(orderNo, responseOrderToken, infoRefund);
            } else {
                var Logger = require('dw/system/Logger');
                Logger.error('OrderToken not received or missing at nemuru response');
            }
        }
    }

    return;
});

/**
 * AddonPayments-PaymentConfirming: The Payment Confirming async response
 * @name AddonPayments-PaymentConfirming
 * @function
 * @memberof AddonPayments
 */
server.get(
    'PaymentConfirming',
    server.middleware.https,
    function (req, res, next) {
        const orderNumber = checkoutHelpers.getOrderNoInSession();
        const orderToken = checkoutHelpers.getOrderToInSession();
        if (orderNumber && orderToken) {
            const order = addonPaymentHelpers.getOrder(orderNumber, orderToken);
            if (order) {
                const addonPaymentInstrument = getAddonPaymentInstrument(order);
                if (addonPaymentInstrument && addonPaymentInstrument.custom) {
                    res.render('checkout/billing/paymentOptions/confirmingPayment', {
                        addonPaymentId : addonPaymentInstrument.custom.addonPaymentID,
                        orderId        :
                        addonPaymentInstrument.custom.addonPaymentsMerchantID.split('_')[0],
                        redirectUrl: getUrls().placeOrderStage
                    });
                }
            }
        }

        return next();
    }
);

/**
 * AddonPayments-PaymentConfirming: The Payment Confirming response
 * @name AddonPayments-PaymentConfirming
 * @function
 * @memberof AddonPayments
 */
server.post('Confirm', server.middleware.https, function (req, res, next) {
    const { addonPaymentId, orderId } = JSON.parse(req.body);

    const addonPaymentsTransaction = getAddonPaymentsNewTransaction(
        orderId,
        addonPaymentId
    );

    if (addonPaymentsTransaction && addonPaymentsTransaction.hasNext()) {
        res.json({ result: true });
    } else {
        res.json({ result: false });
    }

    return next();
});

/**
 * AddonPayments-StatusNemuru: The Nemuru status async response
 * @name AddonPayments-StatusNemuru
 * @function
 * @memberof AddonPayments
 */
server.get('StatusNemuru', server.middleware.https, function (req, res, next) {
    let merchantTransactionId = req.querystring.merchantTransactionId;
    let addonPaymentId = req.querystring.addonPaymentID;
    const addonPaymentsTransaction = getAddonPaymentsNewTransaction(
        merchantTransactionId.split('_')[0],
        addonPaymentId
    );

    if (addonPaymentsTransaction && addonPaymentsTransaction.hasNext()) {
        const addonPaymentsOrdersId = addonPaymentsTransaction.next();
        res.json({
            status      : addonPaymentsOrdersId.custom.addonPaymentStatus,
            redirectUrl : getUrls().placeOrderStage
        });
    } else {
        res.json({ status: 'REDIRECTED' });
    }

    return next();
});

/**
 * AddonPayments-OnlineResponse: The AddonPayments response
 * @name AddonPayments-OnlineResponse
 * @function
 * @memberof AddonPayments
 */
server.get('OnlineResponse', function (req, res, next) {
    const orderNo = checkoutHelpers.getOrderNoInSession();
    const orderTo = addonPaymentHelpers.getOrderTokenInSession();
    const addonPaymentID = session.custom.addonPaymentId;

    if (!orderNo || !orderTo) {
        Logger.warn('Invalid merchant parameters.');
        res.redirect(URLUtils.url('Checkout-Begin', 'stage', 'payment', 'addonPaymentsError', 'true'));
        return next();
    }

    const order = addonPaymentHelpers.getOrder(orderNo, orderTo);
    if (!order) {
        Logger.warn('Not found the order. OrderNo: {0}', orderNo);
        res.redirect(URLUtils.url('Checkout-Begin', 'stage', 'payment', 'addonPaymentsError', 'true'));
        return next();
    }
    
    const addonPaymentsTransaction = getAddonPaymentsNewTransaction(
        orderNo,
        addonPaymentID
    );

    if (addonPaymentsTransaction && addonPaymentsTransaction.hasNext()) {
        const addonPaymentsTransactionId = addonPaymentsTransaction.next();
        const status = addonPaymentsTransactionId && addonPaymentsTransactionId.custom ? addonPaymentsTransactionId.custom.addonPaymentStatus : '';
        const addonPaymentInstrument = getAddonPaymentInstrument(order);
        if (addonPaymentInstrument && addonPaymentInstrument.custom) {
            Transaction.wrap(function () {
                addonPaymentInstrument.custom.addonPaymentStatus = status;
            });
        }
        if (status !== 'SUCCESS') {
            res.redirect(URLUtils.url('AddonPayments-ResponseKo'));
            return next();
        }
        res.redirect(URLUtils.url('AddonPayments-ResponseOk'));
    }
    next();

});

/**
 * AddonPayments-ResponseOK: The valid AddonPayments response
 * @name AddonPayments-ResponseOK
 * @function
 * @memberof AddonPayments
 */
 server.get('ResponseOk', function (req, res, next) {
    const orderNo = checkoutHelpers.getOrderNoInSession();
    const orderTo = addonPaymentHelpers.getOrderTokenInSession();

    if (!orderNo || !orderTo) {
        Logger.warn('Invalid merchant parameters.');
        res.redirect(URLUtils.url('Checkout-Begin', 'stage', 'payment', 'addonPaymentsError', 'true'));
        return next();
    }

    var order = addonPaymentHelpers.getOrder(orderNo, orderTo);
    if (!order) {
        Logger.warn('Not found the order. OrderNo: {0}', orderNo);
        res.redirect(URLUtils.url('Checkout-Begin', 'stage', 'payment', 'addonPaymentsError', 'true'));
        return next();
    }

    var updateOrderResult = addonPaymentHelpers.orderCompleteFields(order);
    if (!updateOrderResult || updateOrderResult.error) {
        Logger.warn('Cannot update the payment status. OrderNo: ', order.orderNo);
        if ('redirectUrl' in updateOrderResult) {
            res.redirect(updateOrderResult.redirectUrl);
        } else {
            res.redirect(URLUtils.url('Checkout-Begin', 'stage', 'payment', 'redsysError', 'true'));
        }
        return next();
    }

    // Delete order number in session
    checkoutHelpers.deleteOrderNoInSession();
    session.custom.addonPaymentId = null;
    
    req.session.privacyCache.set('usingMultiShipping', false);

    res.redirect(URLUtils.url('Order-GetAddonPaymentsConfirm', 'orderNo', order.orderNo, 'orderToken', order.orderToken).toString());

    next();
});

/**
 * AddonPayments-ResponseKO: The invalid AddonPayments response
 * @name AddonPayments-ResponseKO
 * @function
 * @memberof AddonPayments
 */
 server.get('ResponseKo', function (req, res, next) {
    const orderNo = checkoutHelpers.getOrderNoInSession();
    const orderTo = addonPaymentHelpers.getOrderTokenInSession();

    if (!orderNo || !orderTo) {
        Logger.warn('Invalid merchant parameters.');
        res.redirect(URLUtils.url('Checkout-Begin', 'stage', 'payment', 'addonPaymentsError', 'true'));
        return next();
    }

    var order = addonPaymentHelpers.getOrder(orderNo, orderTo);
    if (!order) {
        Logger.warn('Not found the order. OrderNo: {0}', orderNo);
        res.redirect(URLUtils.url('Checkout-Begin', 'stage', 'payment', 'addonPaymentsError', 'true'));
        return next();
    }

    addonPaymentHelpers.setPaymentStatusNotPaid(order);

    // Delete order number in session
    checkoutHelpers.deleteOrderNoInSession();
    session.custom.addonPaymentId = null;

    res.redirect(URLUtils.url('Checkout-Begin', 'stage', 'payment', 'addonPaymentsError', 'true'));

    next();
});

/**
 * AddonPayments-Widget: The widget preferences
 * @name AddonPayments-Widget
 * @function
 * @memberof AddonPayments
 */
 server.post('Widget', function (req, res, next) {
    let preferences = require('*/cartridge/config/addonPaymentsPreferences');
    let ProductMgr = require('dw/catalog/ProductMgr');
    var BasketMgr = require('dw/order/BasketMgr');

    let { productID , amount } = JSON.parse(req.body);

    if (amount) {
        amount = amount * 100;
    } else if (productID) {
        let product = ProductMgr.getProduct(productID);
        amount = product.getPriceModel().getMinPrice().value * 100;
    } else {
        let currentBasket = BasketMgr.getCurrentBasket();
        amount = currentBasket.getTotalGrossPrice() * 100;
    }

    let jsonData = {
        amount: amount,
        product: preferences.widgetFinancing,
        styleColor: preferences.widgetColor,
        styleVariant : preferences.widgetVariant,
        styleBranding : preferences.widgetBranding,
        style : 'width: 100%',
        clientId: '1e7f466b-f96c-4824-b982-1823b1604d71',
        currency: 'EUR',
        locale : 'es-ES',
        decimalSeparator : '.',
        thousandSeparator : ','
    };

    res.json(jsonData);

    return next();

});

module.exports = server.exports();
