'use strict';

const serviceName = 'int_addonPayments.http.rest';
const Resource = require('dw/web/Resource');

const { createErrorLog } = require('../addonPayments/bmAddonPaymentsUtils');

/** createRequest callback for a service
 * @param  {dw.svc.Service} service service instance
 * @param  {Object} data call data with path, method, body for a call or createToken in case of recursive call
 * @returns {string} request body
 */
function createRequest(service, data) {
  const credential = service.configuration.credential;
  const ServiceCredential = require('dw/svc/ServiceCredential');
  if (!(credential instanceof ServiceCredential)) {
    const { msgf } = Resource;
    throw new Error(
      msgf('service.nocredentials', 'addonPaymentserrors', null, serviceName)
    );
  }
  const { path, header, method, params, body } = data;

  service.setURL(path);
  service.addHeader('Accept', 'application/json');

  for (let key in header) {
    service.addHeader(key, header[key]);
  }

  service.setRequestMethod(method || 'POST');

  for (let key in params) {
    service.addParam(key, params[key]);
  }

  return body ? JSON.stringify(body) : '';
}

module.exports = (function () {
  const { msgf } = Resource;
  let restService;
  try {
    restService = require('dw/svc/LocalServiceRegistry').createService(
      serviceName,
      {
        createRequest: createRequest,
        parseResponse: function (_, httpClient) {
          return JSON.parse(httpClient.getText());
        },
        getRequestLogMessage: function (request) {
          return request;
        },
        getResponseLogMessage: function (response) {
          return response.text;
        },
      }
    );
  } catch (error) {
    createErrorLog(
      msgf('service.error', 'addonPaymentserrors', null, serviceName)
    );
    throw new Error(error);
  }

  return {
    call: function (data) {
      let result;
      try {
        result = restService.setThrowOnError().call(data);
      } catch (error) {
        createErrorLog(
          msgf('service.generalerror', 'addonPaymentserrors', null, serviceName)
        );
        throw new Error(error);
      }
      if (result.isOk() & (result.object.status === 'SUCCESS')) {
        return result.object.operationsArray[result.object.operationSize - 1];
      }
      if (result.object.status !== 'SUCCESS') {
        const rest =
          result.object.operationsArray[result.object.operationSize - 1];
        if (rest.status === 'SUCCESS') {
          return rest;
        } else {
          throw new Error(rest.message);
        }
      }
      if (!result.errorMessage) {
        createErrorLog(
          msgf('service.wrongendpoint', 'addonPaymentserrors', null, data.path)
        );
        throw new Error();
      }

      let errorMessage;
      const errorData = JSON.parse(result.message);
      // For type error ex -> {"error", "error_description"}
      if (errorData) {
        errorMessage = errorData;
      }

      throw new Error(errorMessage.toLowerCase());
    },
  };
})();
