'use strict';

const StringUtils = require('dw/util/StringUtils');
const Calendar = require('dw/util/Calendar');
const Order = require('dw/order/Order');
const OrderMgr = require('dw/order/OrderMgr');
const CustomObjectMgr = require('dw/object/CustomObjectMgr');
const PaymentMgr = require('dw/order/PaymentMgr');
const Money = require('dw/value/Money');

const { createErrorLog } = require('./bmAddonPaymentsUtils');

const addonPaymentsApi = require('./bmAddonPaymentsApi');

const allowedProcessorsIds = 'ADDON_PAYMENTS';

/**
 * Returns Addon payment method ID
 * @returns {string} active Addon payment method id
 */
function getAddonPaymentMethodId() {
  const activePaymentMethods = PaymentMgr.getActivePaymentMethods();
  let addonPaymentMethodID;

  activePaymentMethods.toArray().forEach(function (paymentMethod) {
    if (paymentMethod.paymentProcessor.ID === allowedProcessorsIds) {
      addonPaymentMethodID = paymentMethod.ID;
    }
  });
  
  return addonPaymentMethodID;
}

/**
 * Return addon payments order payment instrument
 *
 * @param {dw.order.LineItemCtnr} basket - Basket
 * @returns {dw.order.OrderPaymentInstrument} payment instrument with id addon payments
 */
function getAddonPaymentInstrument(basket) {
  const paymentInstruments = basket.getPaymentInstruments(getAddonPaymentMethodId());
  return !empty(paymentInstruments) && paymentInstruments[0];
}

/**
 *  Update refund transactionID for Addon payment instrument
 *
 * @param {dw.order.Order} order - order
 * @param {object} orderData -  order information
 * @param {dw.order.OrderPaymentInstrument} paymentInstrument instrument with id addon payments
 */
function updateRefundTransaction(order, orderData, paymentInstrument) {
  let paymentRefund = Number(paymentInstrument.custom.addonPaymentRefund.split(' ')[1]);

  paymentRefund += Number(orderData.amount);

  const refund = new Money(paymentRefund, orderData.currency);

  let paymentStatus =
    orderData.operationType === 'REBATE' ? 'PARTIALLY_REFUNDED' : orderData.operationType;

  paymentStatus = order.totalGrossPrice.value === paymentRefund ? 'REFUND' : paymentStatus;

  paymentInstrument.custom.addonPaymentStatus = paymentStatus;

  paymentInstrument.custom.addonPaymentRefund = refund.toString();

  if (
    orderData.operationType === 'REFUND' &&
    Number(paymentInstrument.custom.addonPaymentID) ===
      Number(orderData.originalPayFrexTransactionId)
  ) {
    paymentInstrument.custom.addonPaymentID = orderData.payFrexTransactionId;
  }
  if (paymentStatus === 'REFUND') {
    order.setPaymentStatus(Order.PAYMENT_STATUS_NOTPAID);
    order.setStatus(Order.ORDER_STATUS_CANCELLED);
  }
}

/**
 *  Update void or capture transactionID for addon payment instrument
 *
 * @param {dw.order.Order} order - order
 * @param {object} orderData -  order information
 * @param {dw.order.OrderPaymentInstrument} paymentInstrument instrument with id Addon Payments
 */
function updateVoidCaptureTransaction(order, orderData, paymentInstrument) {
  let paymentStatus = orderData.operationType;

  if (
    Number(paymentInstrument.custom.addonPaymentID) ===
    Number(orderData.originalPayFrexTransactionId)
  ) {
    paymentInstrument.custom.addonPaymentID = orderData.payFrexTransactionId;
  }

  if (paymentStatus === 'VOID') {
    paymentInstrument.custom.addonPaymentStatus = paymentStatus;
    order.setPaymentStatus(Order.PAYMENT_STATUS_NOTPAID);
    order.setStatus(Order.ORDER_STATUS_CANCELLED);
  } else {
    paymentInstrument.custom.addonPaymentStatus = orderData.status;
    order.setPaymentStatus(Order.PAYMENT_STATUS_PAID);
  }
}

/**
 * Update transactionID and transactions history for addon payment instrument
 *
 * @param {string} orderNo - order number
 * @param {object} orderData -  order information
 * @returns {boolean} true in case of success and false when error
 */
function updateOrderTransaction(orderNo, orderData) {
  try {
    const order = OrderMgr.getOrder(orderNo);
    const paymentInstrument = getAddonPaymentInstrument(order);

    if (orderData.operationType === 'REFUND' || orderData.operationType === 'REBATE') {
      updateRefundTransaction(order, orderData, paymentInstrument);
    } else {
      updateVoidCaptureTransaction(order, orderData, paymentInstrument);
    }
  } catch (error) {
    createErrorLog(error);
    return false;
  }

  return true;
}

/**
 * Update transactionID and transactions history for addon payment instrument
 *
 * @param {Object} transaction - order number
 * @param {Object} data -  Indicate if current order is Custom Object
 * @returns {Object} true in case of success and false when error
 */
function getShippingData(transaction, data) {
  transaction.firstname = data.firstName;
  transaction.lastname = data.lastName;
  transaction.shipping = {
    name: { full_name: data.fullName },
    address: {
      address_line_1: data.address1,
      address_line_2: data.address2,
      admin_area_1: data.stateCode,
      admin_area_2: data.city,
      country_code: data.countryCode.value,
      postal_code: data.postalCode,
    },
  };
}

/**
 * Return addon payments order payment instrument
 *
 * @param {Object} order - dw.order.Order
 * @param {boolean} isCustomOrder - isCustomOrder
 * @returns {Object} reorganized data
 */
function prepareOrderTransactionData(order, isCustomOrder) {
  const transaction = {};
  const shippingAmount = !isCustomOrder ? order.shippingTotalPrice.value : 0;
  const taxAmount = !isCustomOrder ? order.totalTax.value : 0;
  let refund;
  let capture;
  const email = order.customerEmail || 'Unknown';
  let addonPaymentInstrument;

  const paymentInstrument = getAddonPaymentInstrument(order);
  addonPaymentInstrument = paymentInstrument.custom;

  transaction.paymentstatus = addonPaymentInstrument.addonPaymentStatus;
  transaction.isCaptureButtonAllowed = true;
  transaction.isCaptured = order.getPaymentStatus().getValue() === order.PAYMENT_STATUS_PAID;

  if (transaction.isCaptured) {
    transaction.isCaptureButtonAllowed = false;
    transaction.capturedAmount = order.totalGrossPrice.value;
    transaction.refundedAmount = !empty(addonPaymentInstrument.addonPaymentRefund)
      ? addonPaymentInstrument.addonPaymentRefund.split(' ')[1]
      : 0.0;
    transaction.restRefountAmount = Number(
      transaction.capturedAmount - transaction.refundedAmount
    ).toFixed(2);
  }

  transaction.transactionid = addonPaymentInstrument.addonPaymentID;
  transaction.orderNo = order.orderNo;
  transaction.amount = order.totalGrossPrice.value;
  transaction.currencycode = order.currencyCode;
  transaction.merchantTransactionId = addonPaymentInstrument.addonPaymentsMerchantID;
  transaction.paymentSolution = addonPaymentInstrument.addonPaymentSolution;
  transaction.orderTimeCreated = order.creationDate ? order.creationDate : '';
  transaction.orderTimeUpdated = order.lastModified ? order.lastModified : '';

  //Add shipping data
  getShippingData(transaction, order.billingAddress);

  return {
    transaction: transaction,
    shippingAmount: shippingAmount,
    taxAmount: taxAmount,
    email: email,
    order: order,
    isCustomOrder: isCustomOrder,
    transactionID: transaction.transactionid,
    isExpiredHonorPeriod: false,
  };
}

module.exports = {
  updateOrderTransaction: updateOrderTransaction,
  getAddonPaymentInstrument: getAddonPaymentInstrument,
  prepareOrderTransactionData: prepareOrderTransactionData
};
