'use strict';

const Logger = require('dw/system/Logger');
const Cipher = require('dw/crypto/Cipher');
const Encoding = require('dw/crypto/Encoding');
const MessageDigest = require('dw/crypto/MessageDigest');
const SecureRandom = require('dw/crypto/SecureRandom');
const Bytes = require('dw/util/Bytes');
const MimeEncodedText = require('dw/value/MimeEncodedText');

let addonPaymentsLogger;
/**
 * Get logger instance
 *
 * @param {msg} msg Error message
 */
function createErrorLog(msg) {
  addonPaymentsLogger =
    addonPaymentsLogger ||
    Logger.getLogger('AddonPayments-BM', 'Addon_Payments_General');
  if (!empty(msg)) {
    addonPaymentsLogger.error(msg);
  } else {
    addonPaymentsLogger.debug('Empty log entry');
  }
  return;
}

function createEncrypt(data) {
  const preferences = require('./bmAddonPaymentsPreferences');
  const pwd = Encoding.toBase64(
    Bytes(preferences.getPreferences().merchantPwd)
  );
  const merchantId = preferences.getPreferences().merchantId;
  const paymentSolution = data.paymentSolution ? data.paymentSolution : null;
  let text = 'merchantId=' + merchantId;
  data
    .entrySet()
    .toArray()
    .forEach(function ({ key, value }) {
      if (
        key === 'amount' ||
        key === 'transactionId' ||
        key === 'merchantTransactionId' ||
        key === 'paymentSolution' ||
        key === 'description'
      ) {
        if( key !== 'amount' && paymentSolution === 'QUIX') {
          if( key === 'paymentSolution'){
            text += '&' + key + '=31600';
          } else {
            text += '&' + key + '=' + value + '';
          }
        } else {
          text += '&' + key + '=0';
        }
      }
    });

    if ( data.orderToken){
      text += '&merchantParams=orderToken:' + data.orderToken;
    }

  // Encode values using UTF-8 charset.
  text = Bytes(text.trim());

  // Hash the unencrypted parameter string using the SHA256 algorithm
  const digest = new MessageDigest(MessageDigest.DIGEST_SHA_256);
  const integrityCheck = digest.digest(Bytes(text));

  //Create Initial Vector
  const initializationVector = Encoding.toBase64(
    new SecureRandom().nextBytes(16)
  );

  const encryptedText = new Cipher().encrypt(
    text,
    pwd,
    'AES/CBC/PKCS5Padding',
    initializationVector,
    0
  );

  var encryptedPayload = {};

  return (
    encryptedPayload = {
      params: {
        encrypted: encryptedText,
        integrityCheck: integrityCheck,
        merchantId: merchantId
      },
      header: {
        encryptionMode: 'CBC',
        iv: initializationVector,
        apiVersion: '3'
      }
    }
  );

}

module.exports = {
  createErrorLog: createErrorLog,
  createEncrypt: createEncrypt,
};
