const addonPaymentsRestService = require('../service/bmAddonPaymentsRestService');

const { createErrorLog, createEncrypt } = require('./bmAddonPaymentsUtils');

const { getPurchaseUnit } = require('./bmPaymentInstrumentHelper');

const util = require('dw/util');
const HttpParameterMap = require('dw/web/HttpParameterMap');
const preferences = require('./bmAddonPaymentsPreferences');

/**
 * Voids, or cancels, an authorized payment, by ID.
 * You cannot void an authorized payment that has been fully captured.
 * NOT Captured Payment !!!
 *
 * TRANSACTIONID(authorization_id) - purchase_units[0].payments.authorizations[0].id
 *
 * @param {Object} data Object with Request Fields
 * @returns {Object} Call handling result
 */
function DoVoid(data) {
  try {
    if (!data.transactionId) {
      createErrorLog('No authorization_id was found during voiding');
      throw new Error();
    }

    const payload = createEncrypt(data);

    const resp = addonPaymentsRestService.call({
      path: preferences.getPreferences().apiRedirectUrl + '/rest/online/void',
      header: payload.header,
      params: payload.params,
    });

    if (resp.status !== 'SUCCESS') {
      createErrorLog(resp.message);
      throw new Error(resp.message);
    }

    resp.responseData = {
      ack: 'Success',
    };
    return resp;
  } catch (err) {
    return {
      err: true,
      responseData: {
        l_longmessage0: err.message,
      },
    };
  }
}

/**
 * Refunds a captured payment, by ID.
 * For a full refund, include an empty payload in the JSON request body.
 * For a partial refund, include an amount object in the JSON request body
 *
 * TRANSACTIONID(capture_id) - purchase_units[0].payments.captures[0].id
 * { "amount": { "value": "999.99", "currency_code": "USD"} } || {}
 *
 * @param {Object} data Object with Request Fields
 * @returns {Object} Call handling result
 */
function RefundTransaction(data) {
  try {
    if (!data.transactionId) {
      createErrorLog('No capture_id was found');
      throw new Error();
    }
    if( data.paymentSolution === 'QUIX'){
      var OrderMgr = require('dw/order/OrderMgr');
      var order = OrderMgr.getOrder(data.orderNo);
      var orderToken = null;
      if (order) {
        orderToken = order.orderToken;
      }
      if (orderToken) {
        data.put('orderToken', orderToken);
      }
    }
    
    const payload = createEncrypt(data);

    const resp = addonPaymentsRestService.call({
      path: preferences.getPreferences().apiRedirectUrl + '/rest/online/rebate',
      header: payload.header,
      params: payload.params,
    });

    var respInfo = null;
    var detailsObj = JSON.parse(resp.details);

    const {
      getUrls
  } = require('../../../../int_addon_payments_sfra/cartridge/scripts/addonPayments/addonPaymentsUtils');

    if(resp.status === 'AWAITING_PAYSOL'){
      respInfo = {
        locale: 'en-ES',
        accessToken: detailsObj.authToken,
        clientId: detailsObj.clientId,
        currency: resp.currency,
        orderId: detailsObj.orderId
      }
    }

    if (resp.status !== 'SUCCESS' && resp.status !== 'AWAITING_PAYSOL') {
      createErrorLog(resp.message);
      throw new Error(resp.message);
    }

    resp.responseData = {
      ack: 'Success',
      respInfo: respInfo
    };
    return resp;
  } catch (err) {
    return {
      err: true,
      responseData: {
        l_longmessage0: err.message,
      },
    };
  }
}

/**
 * Function to capture an authorized payment
 *
 * @param {Object} data Object with Request Fields
 * @returns {Object} Call handling result
 */
function DoCapture(data) {
  try {
    if (!data.transactionId) {
      createErrorLog('No authorization_id was found during capturing');
      throw new Error();
    }

    const payload = createEncrypt(data);

    const resp = addonPaymentsRestService.call({
      path:
        preferences.getPreferences().apiRedirectUrl + '/rest/online/capture',
      header: payload.header,
      params: payload.params,
    });

    if (resp.status !== 'SUCCESS') {
      createErrorLog(resp.message);
      throw new Error(resp.message);
    }

    resp.responseData = {
      ack: 'Success',
    };
    return resp;
  } catch (err) {
    return {
      err: true,
      responseData: {
        l_longmessage0: err.message,
      },
    };
  }
}

/**
 * Call Addon Payments API method
 *
 * @param {string} methodName API method name
 * @param {Object} methodData API method data
 * @returns {Object} Call handling result
 */
function callMethod(methodName, methodData) {
  let data = new util.HashMap();

  let keys;
  if (methodData instanceof HttpParameterMap) {
    keys = methodData.getParameterNames().toArray();
  } else if (methodData instanceof util.HashMap) {
    keys = methodData.keySet().toArray();
  } else {
    keys = Object.keys(methodData);
  }
  keys.forEach(function (property) {
    if (property !== 'methodName' && !empty(methodData[property])) {
      data.put(property, methodData[property].toString());
    }
  });

  try {
    const respData = this[methodName](data);
    return respData;
  } catch (err) {
    return {
      err: err.message,
    };
  }
}

module.exports = {
  DoVoid: DoVoid,
  RefundTransaction: RefundTransaction,
  DoCapture: DoCapture,
  callMethod: callMethod,
};
