/* global dw request response empty */

const paymentInstrumentHelper = require('*/cartridge/scripts/addonPayments/bmPaymentInstrumentHelper');
const {
    createErrorLog
} = require('*/cartridge/scripts/addonPayments/bmAddonPaymentsUtils');
const addonPaymentsApi = require('*/cartridge/scripts/addonPayments/bmAddonPaymentsApi');

const ISML = require('dw/template/ISML');
const Transaction = require('dw/system/Transaction');
const CSRFProtection = require('dw/web/CSRFProtection');
const SystemObjectMgr = require('dw/object/SystemObjectMgr');
const CustomObjectMgr = require('dw/object/CustomObjectMgr');
const ArrayList = require('dw/util/ArrayList');
const Calendar = require('dw/util/Calendar');
const StringUtils = require('dw/util/StringUtils');
const Money = require('dw/value/Money');
const OrderMgr = require('dw/order/OrderMgr');
const PagingModel = require('dw/web/PagingModel');
const Order = require('dw/order/Order');
const PropertyComparator = require('dw/util/PropertyComparator');

/**
 * Get AddonPaymentsNewTransactions Custom Object with given order number
 *
 * @param {string} orderNo - Order number
 * @returns {Object} (transactionIdFromOrder: String - Transaction ID from order, order: dw.object.CustomObject - Custom Object that matched with order number)
 */
function getCustomOrderInfo(orderNo) {
    let order;
    let transactionId;
    try {
        order = CustomObjectMgr.getCustomObject(
            'AddonPaymentsNewTransactions',
            orderNo
        );
        transactionId = order.custom.transactionId;
    } catch (error) {
        createErrorLog(error);
        return false;
    }
    return {
        transactionIdFromOrder : transactionId,
        order                  : order
    };
}

/**
 * Combine orders and AddonPaymentsNewTransactions Custom Objects into one array for pagination
 *
 * @param {string} orderNo - Order number used in "Search by order number" feature
 * @returns {dw.util.ArrayList} Combined array with all orders
 */
function getOrders(orderNo) {
    const systemOrders = SystemObjectMgr.querySystemObjects(
        'Order',
        "orderNo LIKE {0} AND custom.addonPaymentMethod = 'ADDON_PAYMENTS' AND status != {1}",
        'creationDate desc',
        orderNo,
        Order.ORDER_STATUS_FAILED
    );

    let orders = new ArrayList(); // eslint-disable-line no-shadow
    let order;
    let paymentInstrument;
    let orderDate;
    let orderTotal;
    let obj;

    const orderIndex = 0;
    const maxSystemOrdersCount = 9000;

    while (systemOrders.hasNext()) {
        orderIndex++;
        if (orderIndex > maxSystemOrdersCount) {
            break;
        }
        order = systemOrders.next();
        paymentInstrument =
      paymentInstrumentHelper.getAddonPaymentInstrument(order);
        if (paymentInstrument === null) {
            continue; // eslint-disable-line no-continue
        }
        orderDate = new Date(order.creationDate);
        obj = {
            orderNo   : order.orderNo,
            orderDate : StringUtils.formatCalendar(
                new Calendar(orderDate),
                'M/dd/yy h:mm a'
            ),
            createdBy           : order.createdBy,
            isRegestered        : order.customer.registered,
            customer            : order.customerName,
            email               : order.customerEmail,
            orderTotal          : order.totalGrossPrice,
            currencyCode        : order.getCurrencyCode(),
            addonPaymentsAmount : paymentInstrument
                .getPaymentTransaction()
                .getAmount(),
            status      : paymentInstrument.custom.addonPaymentStatus,
            dateCompare : orderDate.getTime(),
            isCustom    : false,
            transactionID       : paymentInstrument.custom.addonPaymentID
        };
        orders.push(obj);
    }

    return orders;
}

/**
 * Render Template
 * @param {string} templateName - Template Name
 * @param {Object} data - pdict data
 */
function render(templateName, data) {
    if (typeof data !== 'object') {
        data = {}; // eslint-disable-line no-param-reassign
    }
    try {
        ISML.renderTemplate(templateName, data);
    } catch (e) {
        throw new Error(e.javaMessage + '\n\r' + e.stack, e.fileName, e.lineNumber);
    }
}

/**
 * Render JSON from Objects
 * @param {Object} responseResult - Response Result
 * @param {Object} responseData - Response Data
 */
function renderJson(responseResult, responseData) {
    var data = {};
    if (!empty(responseData)) {
        data.transactionid = !empty(responseData.transactionid)
            ? responseData.transactionid
            : null;
        data.l_longmessage0 = !empty(responseData.l_longmessage0)
            ? responseData.l_longmessage0
            : null;
        data.ack = !empty(responseData.ack) ? responseData.ack : null;
        data.respInfo = !empty(responseData.respInfo) ? responseData.respInfo : null;
        data.orderToken = !empty(responseData.orderToken) ? responseData.orderToken : null;
    }

    if (!empty(responseResult)) {
        data.result = responseResult;
    }

    response.setContentType('application/json');
    response.writer.print(JSON.stringify(data, null, 2));
}

/**
 * Returns max amount is allowed for multiple capture operation
 */
function helperGetCaptureAmount() {
    var order = null;
    var responseResult = 'Success';

    if (!empty(request.httpParameterMap.orderNo.value)) {
        if (request.httpParameterMap.isCustomOrder.booleanValue) {
            var orderInfo = getCustomOrderInfo(
                request.httpParameterMap.orderNo.stringValue
            );
            if (!orderInfo) {
                responseResult = 'Error';
            } else {
                order = orderInfo.order;
            }
        } else {
            order = OrderMgr.getOrder(request.httpParameterMap.orderNo.stringValue);
        }
    }

    if (!order) {
        responseResult = 'Error';
    }

    renderJson(responseResult);
}

/**
 * Get orders list. Can be filtered by order ID or transaction ID
 */
function orders() {
    let orderNo;
    let alternativeFlow = false;
    let orders; // eslint-disable-line no-shadow
    var transactionId = null;

    if (
        request.httpParameterMap.transactionId.submitted &&
    !empty(request.httpParameterMap.transactionId.stringValue)
    ) {
        transactionId = request.httpParameterMap.transactionId.stringValue;
        const systemOrder = SystemObjectMgr.querySystemObjects(
            'Order',
            "custom.addonPaymentMethod = 'ADDON_PAYMENTS' AND custom.ADDON_PAYMENTS_API_PaymentID LIKE {0} AND status != {1}",
            'creationDate desc',
            transactionId,
            Order.ORDER_STATUS_FAILED
        );
        const addonPaymentsOrder = CustomObjectMgr.queryCustomObjects(
            'AddonPaymentsNewTransactions',
            'custom.addonPaymentId = {0}',
            null,
            transactionId
        );
        if (addonPaymentsOrder.count) {
            orderNo = new ArrayList(addonPaymentsOrder).toArray()[0].custom.orderNo;
        } else if (systemOrder.count) {
            orderNo = new ArrayList(systemOrder).toArray()[0].orderNo;
        }
    }

    if (!orderNo) {
        alternativeFlow = true;
    }

    if (alternativeFlow) {
        orderNo = empty(request.httpParameterMap.orderNo.stringValue)
            ? '*'
            : request.httpParameterMap.orderNo.stringValue;
        orderNo = request.httpParameterMap.transactionId.submitted ? '0' : orderNo;
        orderNo =
      request.httpParameterMap.transactionId.stringValue === '' ? '*' : orderNo;
    }

    try {
        orders = getOrders(orderNo);
    } catch (error) {
        createErrorLog(error);
        render('addonPaymentsbm/components/serverError');
        return;
    }

    var pageSize = !empty(request.httpParameterMap.pagesize.intValue)
        ? request.httpParameterMap.pagesize.intValue
        : 10;
    var currentPage = request.httpParameterMap.page.intValue
        ? request.httpParameterMap.page.intValue
        : 1;
    pageSize = pageSize === 0 ? orders.length : pageSize;
    var start = pageSize * (currentPage - 1);
    var orderPagingModel = new PagingModel(orders);

    orderPagingModel.setPageSize(pageSize);
    orderPagingModel.setStart(start);

    render('addonPaymentsbm/orderList', {
        PagingModel: orderPagingModel
    });
}

/**
 * Get order transaction details
 */
function orderTransaction() {
    let errorFlow = false;
    let order = null;
    let paymentInstrument = null;
    let transactionIdFromOrder = null;
    let paymentInstrumentCustomEl = null;
    let transactionID = null;

    if (
        request.httpParameterMap.orderNo &&
    !empty(request.httpParameterMap.orderNo.value)
    ) {
        if (
            request.httpParameterMap.isCustomOrder &&
      !empty(request.httpParameterMap.isCustomOrder.stringValue)
        ) {
            var orderInfo = getCustomOrderInfo(
                request.httpParameterMap.orderNo.stringValue
            );
            if (!orderInfo) {
                errorFlow = true;
            } else {
                order = orderInfo.order;
                transactionIdFromOrder = orderInfo.transactionIdFromOrder;
            }
        } else {
            order = OrderMgr.getOrder(request.httpParameterMap.orderNo.stringValue);
            if (order) {
                paymentInstrument =
          paymentInstrumentHelper.getAddonPaymentInstrument(order);
            }
        }
    }

    if (!order || (!paymentInstrument && !transactionIdFromOrder)) {
        errorFlow = true;
    }

    if (errorFlow) {
        render('addonPaymentsbm/components/serverError');
        return;
    }

    if (!transactionIdFromOrder) {
        paymentInstrumentCustomEl = paymentInstrument.getCustom();
        transactionIdFromOrder =
      paymentInstrumentCustomEl.addonPaymentID ||
      paymentInstrumentCustomEl.addonPaymentsMerchantID;
    }

    if (
        (!empty(request.httpParameterMap.transactionId.stringValue) &&
      !empty(transactionIdFromOrder) &&
      request.httpParameterMap.transactionId.stringValue !==
        transactionIdFromOrder) ||
    empty(request.httpParameterMap.transactionId.stringValue)
    ) {
        transactionID = transactionIdFromOrder;
    } else {
        transactionID = request.httpParameterMap.transactionId.stringValue;
    }

    const isCustomOrder = !empty(
        request.httpParameterMap.isCustomOrder.stringValue
    );

    const preparedData = paymentInstrumentHelper.prepareOrderTransactionData(
        order,
        isCustomOrder
    );
    render('addonPaymentsbm/orderTransaction', preparedData);
}

/**
 * Do some action, like DoAuthorize, DoCapture, DoRefund and etc
 */
function action() {
    var params = request.httpParameterMap;
    var responseResult = 'Success';
    var callApiResponse = {};
    if (!CSRFProtection.validateRequest()) {
        var errorMsg = {
            l_longmessage0: 'CSRF token mismatch'
        };
        renderJson('Error', errorMsg);
        return;
    }

    if (!params.helperAction.submitted) {
        var methodName = params.methodName.stringValue;
        var methodData = params;
        var orderNo = params.orderNo.stringValue;

        var isCustomOrder = params.isCustomOrder.booleanValue;
        var isSaveToCustomOrder =
      methodName !== 'DoReferenceTransaction' &&
      methodName !== 'DoDirectPayment';

        callApiResponse = addonPaymentsApi.callMethod(methodName, methodData);
        if (callApiResponse.err) {
            responseResult = 'Error';
        } else if (isSaveToCustomOrder && (callApiResponse.paymentSolution !== 'quix' ||
            callApiResponse.operationType !== 'REBATE')) {
        // eslint-disable-line no-lonely-if
            var orderTransactionResult = {};
            Transaction.wrap(function () {
                orderTransactionResult =
            paymentInstrumentHelper.updateOrderTransaction(
                orderNo,
                callApiResponse
            );
            });

            if (!orderTransactionResult) {
                responseResult = 'Error';
            }
        } else {
        // eslint-disable-line no-lonely-if
            if (callApiResponse.paymentSolution !== 'quix' ||
            callApiResponse.operationType !== 'REBATE'){
                var getTransactionResult = addonPaymentsApi.getOrderDetails(
                    callApiResponse.responseData.transactionid
                );
    
                if (getTransactionResult.err) {
                    responseResult = 'Error';
                }
            }
        }
    } else {
        if (params.helperAction.stringValue === 'getCaptureAmount') {
            helperGetCaptureAmount();
            return;
        }
        responseResult = 'Error';
    }

    renderJson(responseResult, callApiResponse.responseData);
}

orders.public = true;
orderTransaction.public = true;
action.public = true;

exports.Orders = orders;
exports.OrderTransaction = orderTransaction;
exports.Action = action;
