<?php

declare(strict_types=1);

namespace ComerciaGlobalPayments\AddonPayments\SDK\Request;

use Symfony\Component\Validator\Constraints as Assert;
use Symfony\Component\Validator\Context\ExecutionContextInterface;

/**
 * Class AccountDisableRequest.
 *
 * Maps data for disable account request.
 *
 * @see https://api-developer.easypaymentgateway.com/EPGJS/#operation/DisableAccount
 */
class AccountDisable extends AbstractRequest
{
    public const AUTH_TOKEN_HEADER = 'authToken';

    /**
     * @var string
     * @Assert\NotNull()
     */
    private $accountId;

    public function getAccountId(): string
    {
        return $this->accountId;
    }

    public function setAccountId(string $accountId): self
    {
        $this->accountId = $accountId;

        return $this;
    }

    public function setAuthToken(string $authToken): self
    {
        $this->addHeader(self::AUTH_TOKEN_HEADER, $authToken);

        return $this;
    }

    public function getAuthToken(): ?string
    {
        return $this->getAdditionalHeaders()[self::AUTH_TOKEN_HEADER] ?? null;
    }

    /**
     * @Assert\Callback()
     */
    public function validate(ExecutionContextInterface $context, $payload): void
    {
        $token = $this->getAuthToken();
        if (empty($token)) {
            $context->buildViolation('The authorization token is missing')
                ->atPath('additionalHeaders')
                ->addViolation();
        }
    }
}
