<?php

declare(strict_types=1);

namespace ComerciaGlobalPayments\AddonPayments\SDK\Request;

use Symfony\Component\Validator\Constraints as Assert;

/**
 * Class Transactions.
 *
 * Maps data for 'transactions' request.
 *
 * @see https://api-developer.easypaymentgateway.com/EPGJS/#operation/Transaction%20Status
 *
 * @SuppressWarnings(PHPMD.LongVariableNames)
 */
class Transactions extends AbstractEncryptedRequest
{
    /**
     * @var string
     * @Assert\NotNull()
     */
    private $token;

    /**
     * @var array
     * @Assert\NotNull()
     */
    private $transactions = [];

    /**
     * @var int
     * @Assert\NotNull()
     * @Assert\Range(min = 0, max = 1, notInRangeMessage = "The value {{ value }} is not a valid postback value")
     */
    private $postback = 0;

    /**
     * @var string
     */
    private $url;

    public static function excludedProperties(): array
    {
        return array_merge(parent::excludedProperties(), [
          'transactions',
        ]);
    }

    public function jsonSerialize(): array
    {
        return array_merge(parent::jsonSerialize(), [
          'transactions' => implode(';', $this->transactions),
        ]);
    }

    public function setToken(string $token): self
    {
        $this->token = $token;

        return $this;
    }

    public function setTransactions(array $transactions): self
    {
        $this->transactions = $transactions;

        return $this;
    }

    public function addTransaction(int $transaction): self
    {
        $this->transactions[] = $transaction;

        return $this;
    }

    public function setPostback(int $postback): self
    {
        $this->postback = $postback;

        return $this;
    }

    public function setUrl(string $url): self
    {
        $this->url = $url;

        return $this;
    }

    public function getToken(): string
    {
        return $this->token;
    }

    public function getTransactions(): ?array
    {
        return $this->transactions;
    }

    public function getPostback(): int
    {
        return $this->postback;
    }

    public function getUrl(): string
    {
        return $this->url;
    }
}
